<?php

namespace Mtc\Members\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Redirect;
use Mtc\Foundation\Country;
use Mtc\Members\Facades\MemberAuth;
use Mtc\Members\HasMembersNavigation;
use Mtc\Members\Http\Requests\MemberAddressRequest;
use Mtc\Members\Http\Requests\SaveMemberAddressRequest;
use Mtc\Members\Member;
use Mtc\Members\MemberAddress;

class AddressController extends Controller
{
    use HasMembersNavigation;

    /**
     * Show details
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        /** @var Member $member */
        $member = MemberAuth::user();
        $member->load('addresses.country');

        $this->page_meta['title'] = 'Address Book | ' . config('app.name');
        $this->page_meta['page_title'] = 'Address Book';

        return template('members/addresses/index.twig', [
            'page_meta' => $this->page_meta,
            'member' => $member,
            'sections' => $this->sections(),
        ]);
    }

    /**
     * Create record
     *
     * @param MemberAddressRequest $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create(MemberAddressRequest $request)
    {
        $this->page_meta['title'] = 'Address Book | ' . config('app.name');
        $this->page_meta['page_title'] = 'Add Address';

        return template('members/addresses/edit.twig', [
            'page_meta' => $this->page_meta,
            'address' => $request->address(),
            'countries' => Country::query()->pluck('name', 'code'),
            'sections' => $this->sections(),
        ]);
    }

    /**
     * Store record
     *
     * @param SaveMemberAddressRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(SaveMemberAddressRequest $request)
    {
        /** @var Member $member */
        $member = MemberAuth::user();
        $address = $member->addresses()->create($request->input());

        if ($request->has('set_default') || $member->addresses()->count() == 0) {
            $member->setDefaultAddress($address->id);
        }

        return Redirect::to(route('members.addresses.index'))->with('success', 'Address Saved');
    }

    /**
     * Edit a record
     *
     * @param MemberAddressRequest $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(MemberAddressRequest $request)
    {
        $this->page_meta['title'] = 'Address Book | ' . config('app.name');
        $this->page_meta['page_title'] = 'Manage My Address';

        return template('members/addresses/edit.twig', [
            'page_meta' => $this->page_meta,
            'address' => $request->address(),
            'countries' => Country::query()->pluck('name', 'code'),
            'sections' => $this->sections(),
        ]);
    }

    /**
     * Save record
     *
     * @param SaveMemberAddressRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(SaveMemberAddressRequest $request)
    {
        $request->address()->fill($request->input())->save();
        if ($request->has('set_default')) {
            MemberAuth::user()->setDefaultAddress($request->address()->id);
        }

        return Redirect::to(route('members.addresses.index'))->with('success', 'Address Saved');
    }

    /**
     * Destroy record
     *
     * @param MemberAddressRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(MemberAddressRequest $request)
    {
        $request->address()->delete();
        return Redirect::to(route('members.addresses.index'))->with('success', 'Address Removed');
    }

    /**
     * Set default address
     *
     * @param MemberAddress $address
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|void
     */
    public function setDefault(MemberAddressRequest $request)
    {
        /** @var Member $member */
        $member = MemberAuth::user();
        $member->setDefaultAddress($request->address()->id);

        return Redirect::to(route('members.addresses.index'))->with('success', 'Default Address Updated');
    }
}
