<?php

namespace Mtc\Members;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Members\Contracts\MemberModel;
use Spatie\Permission\Traits\HasRoles;

/**
 * Class Member
 *
 * @package Mtc\Members
 */
class Member extends Authenticatable implements MemberModel
{
    use HasRoles;

    /**
     * This is used for legacy flow
     */
    public const SESSION_KEY = 'member_id';

    /**
     * Guard name for role management
     *
     * @var string
     */
    public $guard_name = 'member';

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'members';

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'email',
        'name',
        'contact_no',
        'mobile_no',
        'approved',
        'avatar',
        'password',
        'default_address_id',
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'hash',
        'remember_token',
    ];

    /**
     * Boot model
     */
    protected static function boot()
    {
        parent::boot();

        self::deleting(function (self $member) {
            $member->addresses()->delete();
            $member->tokens()->delete();
        });
    }

    /**
     * Relationship to all addresses
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function addresses()
    {
        return $this->hasMany(MemberAddress::class);
    }

    /**
     * Default address relationship
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function defaultAddress()
    {
        return $this->belongsTo(MemberAddress::class, 'default_address_id');
    }

    /**
     * Access or assignment tokens for user
     * e.g. stripe customer token for billing
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function tokens()
    {
        return $this->hasMany(MemberTokens::class);
    }

    /**
     * Search filters
     *
     * @param Request $request
     * @return Builder
     */
    public function scopeSearch(Builder $query, Request $request)
    {
        collect(config('members.admin_search_filters', []))
            ->each(function ($filter_class) use ($request, $query) {
                App::make($filter_class)->handle($request, $query);
            });

        return $query;
    }

    /**
     * Set default address for member
     *
     * @param $address_id
     */
    public function setDefaultAddress($address_id)
    {
        $this->default_address_id = $address_id;
        $this->save();
    }
}
