<?php

namespace Mtc\Members;

use Illuminate\Database\Eloquent\Model;
use Mtc\Foundation\Country;

/**
 * Class MemberAddress
 *
 * @package Mtc\Members
 */
class MemberAddress extends Model
{
    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'member_addresses';

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'member_id',
        'name',
        'first_name',
        'last_name',
        'address1',
        'address2',
        'city',
        'postcode',
        'country_code',
        // Additional fields for optional usage
        'address3',
        'state',
        'county',
    ];

    /**
     * Fields that identify address (excluding name)
     *
     * @var array
     */
    public static $address_fields = [
        'address1',
        'address2',
        'city',
        'county',
        'state',
        'postcode',
        'country_code',
    ];
    /**
     * Append values to JSON output
     *
     * @var array
     */
    protected $appends = [
        'address_book_entry',
    ];

    /**
     * Get all filled address fields
     *
     * @return \Illuminate\Support\Collection
     */
    public function getFullAddressAttribute()
    {
        return collect($this->only(self::$address_fields))->filter()
            ->map(fn($value, $key) => $key === 'country' ? $this->formatCountryAttributeValue() : $value)
            ->reject(fn($value) => $value === 'NA');
    }

    /**
     * Relationship to owner of this address
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function member()
    {
        return $this->belongsTo(Member::class);
    }

    /**
     * Relationship with country
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function country()
    {
        return $this->belongsTo(Country::class, 'country_code', 'code');
    }

    /**
     * Get the country name
     *
     * @return string
     */
    public function getCountryNameAttribute()
    {
        return $this->country->name ?? '';
    }

    /**
     * route_hash attribute for masking id on front-end
     *
     * @return string
     */
    public function getRouteHashAttribute()
    {
        return base64_encode($this->id);
    }

    /**
     * attribute that is used in select list
     *
     * @return string
     */
    public function getAddressBookEntryAttribute()
    {
        return $this->first_name . ' ' . $this->last_name . ', ' . $this->getFullAddressAttribute()->implode(', ');
    }

    /**
     * Get country name value
     *
     * @return string
     */
    private function formatCountryAttributeValue()
    {
        return config('foundation.load_full_country_names') === true && !empty($this->country->name)
            ? $this->country->name
            : $this->country_code;
    }
}
