<?php

namespace Mtc\Members\Http\Controllers\Admin;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Redirect;
use Mtc\Members\Facades\MemberAuth;
use Mtc\Members\Http\Requests\AdminMemberRequest;
use Mtc\Members\Http\Requests\SaveAdminMemberRequest;
use Mtc\Members\Role;

/**
 * Class MemberController
 *
 * @package Mtc\Members\Http\Controllers\Admin
 */
class MemberController extends Controller
{
    /**
     * List records
     *
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(AdminMemberRequest $request)
    {
        $this->page_meta['title'] = 'Manage Members';
        $members = $request->member()
            ->newQuery()
            ->search($request)
            ->latest('id')
            ->paginate(50)
            ->appends($request->input());

        return template('members/admin/index.twig', [
            'page_meta' => $this->page_meta,
            'members' => $members,
        ]);
    }

    /**
     * Create new record
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create(AdminMemberRequest $request)
    {
        $this->page_meta['title'] = 'Create Member Record';
        return template('members/admin/edit.twig', [
            'page_meta' => $this->page_meta,
            'member' => $request->member(),
            'roles' => Role::query()->pluck('name', 'id'),
        ]);
    }

    /**
     * Store record
     *
     * @param SaveAdminMemberRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(SaveAdminMemberRequest $request)
    {
        $request->member()->fill($request->input())->save();
        if ($request->has('role')) {
            $request->member()->syncRoles($request->input('role'));
        }
        return Redirect::to(route('admin.members.edit', $request->member()))
            ->with('success', 'Member Saved');
    }

    /**
     * Edit a record
     *
     * @param AdminMemberRequest $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(AdminMemberRequest $request)
    {
        $this->page_meta['title'] = 'Manage Member Record - ' . $request->member()->name;
        return template('members/admin/edit.twig', [
            'page_meta' => $this->page_meta,
            'member' => $request->member(),
            'roles' => Role::query()->pluck('name', 'id'),
        ]);
    }

    /**
     * Update members details
     *
     * @param SaveAdminMemberRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(SaveAdminMemberRequest $request)
    {
        $request->member()->fill($request->input())->save();
        if ($request->has('role')) {
            $request->member()->syncRoles($request->input('role'));
        }
        return Redirect::back()->with('success', 'Member Updated');
    }

    /**
     * Remove record
     *
     * @param $member_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(AdminMemberRequest $request)
    {
        $request->member()->delete();
        return Redirect::back()->with('success', 'Member Record Removed');
    }

    /**
     * Login as member
     *
     */
    public function loginAs(AdminMemberRequest $request)
    {
        MemberAuth::login($request->member());
        return redirect(route('members.dashboard'));
    }
}
