<?php

namespace Mtc\MercuryDataModels;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;

/**
 * @property ?int $parent_id
 * @property ?Carbon $next_due_at
 * @property ?Carbon $last_failed_at
 * @property ?Carbon $last_processed_at
 * @property boolean $is_active
 * @property boolean $is_ready
 * @property boolean $is_running
 * @property string $name
 * @property string $provider
 * @property string $type
 * @property string $schedule
 * @property string $schedule_hours
 */
class ScheduledTask extends Model
{
    protected $fillable = [
        'parent_id',
        'name',
        'is_active',
        'is_ready',
        'is_running',
        'last_processed_at',
        'last_failed_at',
        'next_due_at',
        'schedule',
        'schedule_hours',
        'type',
        'provider',
        'data',
    ];

    protected $casts = [
        'data' => 'array',
        'last_processed_at' => 'datetime',
        'last_failed_at' => 'datetime',
        'next_due_at' => 'datetime',
        'is_active' => 'boolean',
        'is_ready' => 'boolean',
        'is_running' => 'boolean',
    ];

    public function subTasks(): HasMany
    {
        return $this->hasMany(self::class, 'parent_id');
    }

    public function parent(): BelongsTo
    {
        return $this->belongsTo(self::class, 'parent_id');
    }

    public function logs(): HasMany
    {
        return $this->hasMany(ScheduledTaskLog::class, 'task_id')->latest();
    }

    public function getNextDueAt(Carbon $from)
    {
        return App::make(Config::get('reports.schedule_types.' . $this->schedule))
            ->nextTime($from, null, ['hours' => $this->schedule_hours]);
    }

    public function lockTask(): void
    {
        $this->update([
            'is_running' => true,
        ]);
        $this->subTasks()->update([
            'is_ready' => false,
        ]);
    }

    public function unlockTask(bool $was_success = true): void
    {
        $this->update([
            'is_running' => false,
            'is_ready' => false,
            $was_success ? 'last_processed_at' :'last_failed_at' => Carbon::now(),
            'next_due_at' => $this->getNextDueAt(Carbon::now()),
        ]);

        $this->subTasks()->update([
            'is_ready' => true,
        ]);
    }
}
