<?php

namespace Mtc\MercuryDataModels;

use App\Master\Models\VehicleModel as MasterVehicleModel;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Mtc\Crm\Traits\ModelSortAndFilter;
use Mtc\Filter\FilterIndex;
use Mtc\MercuryDataModels\Factories\VehicleModelFactory;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Mtc\MercuryDataModels\Traits\OrderByName;

class VehicleModel extends Model
{
    use HasFactory;
    use OrderByName;
    use ModelSortAndFilter;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'vehicle_models';

    /**
     * Mass assign columns
     *
     * @var string[]
     */
    protected $fillable = [
        'id',
        'make_id',
        'name',
        'autotrader_id',
        'cap_id',
        'lcv_cap_id',
    ];

    protected $appends = [
        'slug'
    ];

    protected $with = [
        'filterIndex'
    ];

    /**
     * Model factory
     *
     * @return VehicleModelFactory
     */
    protected static function newFactory()
    {
        return VehicleModelFactory::new();
    }

    /**
     * Relationship with vehicle make this model belongs
     *
     * @return BelongsTo
     */
    public function make(): BelongsTo
    {
        return $this->belongsTo(VehicleMake::class);
    }

    /**
     * ->ofMake($make_id) to specify model make
     *
     * @param Builder $query
     * @param int $make_id
     * @return Builder
     */
    public function scopeOfMake(Builder $query, $make_id)
    {
        return $query->where('make_id', $make_id);
    }

    /**
     * Relationship with vehicles
     *
     * @return HasMany
     */
    public function vehicles(): HasMany
    {
        return $this->hasMany(Vehicle::class, 'model_id');
    }

    /**
     * Relationship with offers
     *
     * @return HasMany
     */
    public function offers(): HasMany
    {
        return $this->hasMany(VehicleOffer::class, 'model_id');
    }

    /**
     * Relationship with new cars
     *
     * @return HasMany
     */
    public function newCars(): HasMany
    {
        return $this->hasMany(NewCar::class, 'model_id');
    }

    /**
     * Relationship with cheapest offer
     *
     * @return HasOne
     */
    public function cheapestActiveOffer(): HasOne
    {
        return $this->hasOne(VehicleOffer::class, 'model_id')
            ->withListingRelationships()
            ->orderBy('price');
    }

    public function filterIndex()
    {
        return $this->morphOne(FilterIndex::class, 'filter');
    }

    protected function getSlugAttribute(): string
    {
        return $this->filterIndex?->slug ?? '';
    }
}
