<?php

namespace Mtc\MercuryDataModels\Filters;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleAttribute;

/**
 * @property VehicleAttribute $vehicleAttribute
 */
class VehicleAttributeBooleanFilter extends IsFilter implements CustomPatternFilter
{
    protected int $min_count_to_show = 1;

    /**
     * Constructor to set the slug.
     *
     * @param string $type
     */
    public function __construct(protected string $type)
    {
        $this->vehicleAttribute = VehicleAttribute::query()
            ->where('filterable', 1)
            ->where('slug', $this->type)
            ->first();
    }

    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selections = [])
    {
        $query->where(function ($query) use ($selections) {
            foreach ($selections as $selection) {
                if (strtolower($selection) === 'yes') {
                    $query->orWhereHas('attributeValues', function ($query) {
                        $query->where('slug', $this->type)
                            ->where('value_integer', 1);
                    });
                } else {
                    $query->orWhereDoesntHave('attributeValues', function ($query) {
                        $query->where('slug', $this->type)
                            ->where('value_integer', 1);
                    });
                }
            }
        });
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return collect([
            [
                'id' => 'yes',
                'name' => 'Yes',
                'count' => null,
                'slug' => 'yes',
            ],
            [
                'id' => 'no',
                'name' => 'No',
                'count' => null,
                'slug' => 'no',
            ],
        ]);
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'has_load_more' => $this->hasLoadMore($collection),
            'results' => $collection,
            'hide_in_ui' => false,
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
            'is_range_maximum' => $this->isRangeMaximum(),
            'range_group' => $this->getRangeGroup()
        ];
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return Vehicle::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return $this->vehicleAttribute->name;
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return '';
    }

    /**
     * Specify attribute on object that represents id
     *
     * @return string
     */
    public function getIdAttribute(bool $for_index = false): string
    {
        return 'vehicle_attribute_' . $this->type;
    }

    /**
     * Specify attribute on object that represents name
     *
     * @return string
     */
    public function getNameAttribute(bool $for_index = false): string
    {
        return 'vehicle_attribute_' . $this->type;
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/' . $this->type . '-(yes|no)/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        return preg_match('/' . $this->type . '-(yes|no)/', $selection);
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return $this->type . '-' . $selection;
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return $this->vehicleAttribute->name . ': ' . ucfirst($selection);
    }
}
