<?php

namespace Mtc\MercuryDataModels;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Mtc\ContentManager\Contracts\PageModel;
use Mtc\ContentManager\Models\PageUser;
use Mtc\ContentManager\PageStatus;
use Mtc\ContentManager\Traits\HasMedia;
use Mtc\MercuryDataModels\Contracts\ModelWithContent;
use Mtc\MercuryDataModels\Contracts\ModelWithUrlPath;
use Mtc\MercuryDataModels\Contracts\SearchableModel;
use Mtc\MercuryDataModels\Factories\PageFactory;
use Mtc\MercuryDataModels\Tools\UiUrlGenerator;
use OwenIt\Auditing\Contracts\Auditable;

class Page extends PageModel implements SearchableModel, ModelWithUrlPath, Auditable, ModelWithContent
{
    use \OwenIt\Auditing\Auditable;
    use HasMedia;

    protected $hidden = [
        'password',
    ];

    public function getOwnerColumn(): string
    {
        return 'page_id';
    }

    /**
     * Relationship with users that are involved in a page
     *
     * @return BelongsToMany
     */
    public function pageUsers(): HasMany
    {
        return $this->hasMany(PageUser::class);
    }

    public function franchise(): BelongsTo
    {
        return $this->belongsTo(Franchise::class, 'franchise_id');
    }

    /**
     * @param Builder $query
     * @return void
     */
    public function scopeAssignedToMe(Builder $query)
    {
        $query->whereHas('pageUsers', fn ($query) => $query->where('user_id', Auth::id()));
    }

    /**
     * Model factory
     *
     * @return \Mtc\ContentManager\Factories\PageFactory|PageFactory
     */
    protected static function newFactory()
    {
        return PageFactory::new();
    }

    /**
     * Search name text
     *
     * @return string
     */
    public function getSearchNameAttribute(): string
    {
        return $this->title;
    }

    /**
     * Search excerpt text
     *
     * @return string
     */
    public function getSearchExcerptAttribute(): string
    {
        $status = $this->status instanceof PageStatus
            ? $this->status->value
            : $this->status;
        return collect([
            ucwords($status),
            $this->urlPath(),
            $this->categories()->pluck('name')->implode(','),
        ])->filter()->implode(' | ');
    }

    /**
     * Search result icon
     *
     * @return string
     */
    public function getSearchIcon(): string
    {
        return 'file-lines';
    }

    /**
     * Route to viewing a vehicle as a part of search response
     *
     * @return string
     */
    public function getSearchResultRoute(): string
    {
        return UiUrlGenerator::make('manage-content/content/edit/' . $this->id, [], false);
    }

    /**
     * Path to the front-end url
     *
     * @return string
     */
    public function urlPath(): string
    {
        $prefix = !empty($page->category) ? $page->category . '/' : '';
        return '/' . $prefix . $this->slug;
    }

    /**
     * Link with menu entries that have this page
     *
     * @return MorphMany
     */
    public function linkedMenuEntries(): MorphMany
    {
        return $this->morphMany(MenuEntry::class, 'item');
    }


    /**
     * Set filters based on terms passed
     *
     * @param Builder $query
     * @param array|string $filters
     * @return void
     */
    public function scopeSetFilters(Builder $query, array|string $filters)
    {
        collect($filters)
            ->each(function ($filter) use ($query) {
                if (str_starts_with($filter, 'type-')) {
                    $query->where('category', str_replace('type-', '', $filter));
                } elseif (str_starts_with($filter, 'franchise-')) {
                    $query->whereHas(
                        'franchise',
                        fn($query) => $query->where('slug', str_replace('franchise-', '', $filter))
                    );
                } elseif (method_exists($this, 'scope' . ucfirst(Str::camel($filter)))) {
                    $scopeMethod = 'scope' . ucfirst(Str::camel($filter));
                    $this->{$scopeMethod}($query);
                }
            });
    }
}
