<?php

namespace Mtc\MercuryDataModels;

use App\Facades\Settings;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Mtc\ContentManager\Contracts\MediaUse;
use Mtc\ContentManager\Traits\HasMedia;
use Mtc\ContentManager\Traits\ModelSortAndFilter;
use Mtc\MercuryDataModels\Contracts\ModelWithUrlPath;
use Mtc\MercuryDataModels\Contracts\SearchableModel;
use Mtc\MercuryDataModels\Factories\VehicleFactory;
use Mtc\MercuryDataModels\Tools\UiUrlGenerator;
use Mtc\MercuryDataModels\Traits\FormatAsCurrency;
use Mtc\MercuryDataModels\Traits\HasVehicleCustomAttributes;
use Mtc\MercuryDataModels\Traits\VehicleAgeIdentifierFinder;
use Mtc\VehicleReservations\Reservation;
use OwenIt\Auditing\Contracts\Auditable;

/**
 * @property int $id
 * @property string $slug
 * @property string $uuid
 * @property string $vin
 * @property string $type
 * @property boolean $featured
 * @property boolean $was_recently_synced
 * @property string $auto_trader_id
 * @property string $cap_id
 * @property string $motor_check_id
 * @property string $title
 * @property string $derivative
 * @property string $registration_number
 * @property bool $is_new
 * @property int $make_id
 * @property int $model_id
 * @property int $dealership_id
 * @property int $transmission_id
 * @property int $fuel_type_id
 * @property int $drivetrain_id
 * @property string $colour
 * @property float $price
 * @property float|null $original_price
 * @property float|null $previous_price
 * @property float $monthly_price
 * @property float|null $rrp_price
 * @property float|null $deposit
 * @property int $co2
 * @property double $mpg
 * @property int $door_count
 * @property int $manufacture_year
 * @property int $odometer_km
 * @property int $odometer_mi
 * @property int $engine_size_cc
 * @property int $previous_owner_count
 * @property ?Carbon $first_registration_date
 * @property string $description
 * @property array $data
 * @property string $exterior_video_url
 * @property string $interior_video_url
 * @property Carbon $date_in_stock
 */
class Vehicle extends Model implements SearchableModel, ModelWithUrlPath, Auditable
{
    use \OwenIt\Auditing\Auditable;
    use HasFactory;
    use HasMedia;
    use SoftDeletes;
    use ModelSortAndFilter;
    use VehicleAgeIdentifierFinder;
    use FormatAsCurrency;
    use HasVehicleCustomAttributes;

    public const KM_IN_MILES = 1.609344;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'vehicles';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'stock_provider',
        'uuid',
        'vin',
        'auto_trader_id',
        'cap_id',
        'motor_check_id',
        'type',
        'featured',
        'published_at',
        'available_date',
        'is_published',
        'archived_at',
        'finance_exported_at',
        'stock_arrival_date',
        'is_vat_applicable',
        'is_sold',
        'is_demo',
        'is_reserved',
        'title',
        'derivative',
        'registration_number',
        'vrm_condensed',
        'is_new',
        'make_id',
        'model_id',
        'body_style_id',
        'dealership_id',
        'transmission_id',
        'fuel_type_id',
        'drivetrain_id',
        'colour',
        'price',
        'admin_fee',
        'original_price',
        'previous_price',
        'monthly_price',
        'monthly_cost_type',
        'rrp_price',
        'deposit',
        'door_count',
        'seats',
        'manufacture_year',
        'odometer_km',
        'odometer_mi',
        'engine_size_cc',
        'co2',
        'mpg',
        'previous_owner_count',
        'first_registration_date',
        'description',
        'attention_grabber',
        'trim',
        'data',
        'vehicle_length',
        'main_video_url',
        'exterior_video_url',
        'interior_video_url',
        'battery_range',
        'battery_capacity_kwh',
        'battery_usable_capacity_kwh',
        'battery_charge_time',
        'battery_quick_charge_time',
        'battery_quick_charge_level',
        'wheelbase_type',
        'wheelbase_mm',
        'payload_kg',
        'gross_vehicle_weight_kg',
        'bhp',
        'was_recently_synced',
        'pending_stock_sync',
        'lat',
        'lng',
    ];

    /**
     * Cast attributes to types
     *
     * @var string[]
     */
    protected $casts = [
//        'type' => VehicleType::class,  // need L9
        'is_published' => 'boolean',
        'is_new' => 'boolean',
        'is_sold' => 'boolean',
        'is_reserved' => 'boolean',
        'is_demo' => 'boolean',
        'is_vat_applicable' => 'boolean',
        'archived_at' => 'datetime',
        'featured' => 'boolean',
        'was_recently_synced' => 'boolean',
        'data' => 'array',
        'created_at' => 'datetime',
        'first_registration_date' => 'datetime:Y-m-d H:i:s',
        'published_at' => 'datetime:Y-m-d H:i:s',
        'available_date' => 'datetime:Y-m-d H:i:s',
        'date_in_stock' => 'datetime:Y-m-d H:i:s',
        'stock_arrival_date' => 'datetime:Y-m-d',
    ];

    protected $appends = [
        'status',
    ];

    protected static function newFactory()
    {
        return VehicleFactory::new();
    }

    /**
     * Relationship with finance examples
     *
     * @return HasMany
     */
    public function financeExamples(): HasMany
    {
        return $this->hasMany(VehicleFinance::class);
    }

    /**
     * @return VehicleFinance
     */
    public function defaultFinanceExample(): HasOne
    {
        return $this->hasOne(VehicleFinance::class)->orderBy('monthly_price');
    }

    /**
     * Relationship with dealership
     *
     * @return BelongsTo
     */
    public function dealership(): BelongsTo
    {
        return $this->belongsTo(Dealership::class);
    }

    /**
     * Relationship with specs
     *
     * @return MorphMany
     */
    public function specs(): MorphMany
    {
        return $this->morphMany(VehicleTechnicalData::class, 'vehicle');
    }

    public function features(): MorphMany
    {
        return $this->morphMany(VehicleFeature::class, 'vehicle')->orderBy('name');
    }

    public function equipment(): MorphMany
    {
        return $this->morphMany(VehicleStandardEquipment::class, 'vehicle');
    }

    public function labels(): BelongsToMany
    {
        return $this->belongsToMany(Label::class, 'vehicle_labels');
    }

    /**
     * Relationship with model
     *
     * @return BelongsTo
     */
    public function make(): BelongsTo
    {
        return $this->belongsTo(VehicleMake::class, 'make_id');
    }

    /**
     * Relationship with model
     *
     * @return BelongsTo
     */
    public function model(): BelongsTo
    {
        return $this->belongsTo(VehicleModel::class, 'model_id');
    }

    /**
     * Relationship with drivetrain
     *
     * @return BelongsTo
     */
    public function drivetrain(): BelongsTo
    {
        return $this->belongsTo(DrivetrainType::class, 'drivetrain_id');
    }

    /**
     * Relationship with fuel type
     *
     * @return BelongsTo
     */
    public function fuelType(): BelongsTo
    {
        return $this->belongsTo(FuelType::class, 'fuel_type_id');
    }

    /**
     * Relationship with transmission
     *
     * @return BelongsTo
     */
    public function transmission(): BelongsTo
    {
        return $this->belongsTo(TransmissionType::class, 'transmission_id');
    }

    /**
     * Relationship with bodyStyle
     *
     * @return BelongsTo
     */
    public function bodyStyle(): BelongsTo
    {
        return $this->belongsTo(BodyStyleType::class, 'body_style_id');
    }

    /**
     * All enquiries submitted for this vehicle
     *
     * @return MorphMany
     */
    public function allEnquiries(): MorphMany
    {
        return $this->morphMany(config('crm.enquiry_model'), 'reason');
    }

    /**
     * Enquiries submitted for this vehicle in the last 7 days
     *
     * @return MorphMany
     */
    public function last7DaysEnquiries(): MorphMany
    {
        return $this->morphMany(config('crm.enquiry_model'), 'reason')
            ->where('created_at', '>=', Carbon::now()->subDays(7));
    }

    /**
     * Enquiries submitted for this vehicle in current month
     *
     * @return MorphMany
     */
    public function currentMonthEnquiries(): MorphMany
    {
        return $this->morphMany(config('crm.enquiry_model'), 'reason')
            ->where('created_at', '>=', Carbon::now()->startOfMonth());
    }

    public function colourMap(): BelongsTo
    {
        return $this->belongsTo(ColourMap::class, 'colour', 'sub_colour');
    }

    /**
     * Relationship with page views on a particular date
     *
     * @return HasMany
     */
    public function views(): HasMany
    {
        return $this->hasMany(VehicleView::class);
    }

    public function conversions(): MorphMany
    {
        return $this->morphMany(Conversion::class, 'owner');
    }

    /**
     * Relationship with AutoTrader data for vehicle
     *
     * @return HasOne
     */
    public function autoTraderData(): HasOne
    {
        return $this->hasOne(VehicleAutoTraderData::class);
    }

    public function reservations(): HasMany
    {
        return $this->hasMany(Reservation::class, 'vehicle_id');
    }

    /**
     * active() scope
     *
     * @param Builder $query
     * @return void
     */
    public function scopeActive(Builder $query): void
    {
        $query->where(fn ($query) => $query->where('is_published', 1));
    }

    public function scopeWithRelationshipsForCardView(Builder $query): void
    {
        $query
            ->with([
                'make',
                'model',
                'fuelType',
                'transmission',
                'bodyStyle',
                'primaryMediaUse.media',
                'attributeValues',
                'features',
                'dealership.franchise',
                'dealership.primaryDepartment'
            ])
            ->when(
                Settings::get('vehicle-list-load-finance-example') &&
                Settings::get('vehicle-list-load-finance-example-type', 'cheapest') !== 'cheapest',
                fn($query) => $query->with('financeExamples')
            )
            ->when(
                Settings::get('autotrader-advert-performance'),
                fn($query) => $query->with('autoTraderData')
            )
            ->when(
                Settings::get('vehicle-card-image-count') > 1,
                fn($query) => $query->with('mediaUses.media')
            )
            ->when(
                Settings::get('vehicle-list-load-finance-example'),
                fn($query) => $query->with('defaultFinanceExample')
            )
            ->when(
                Settings::get('vehicle-labels-enabled'),
                fn($query) => $query->with('labels')
            );
    }

    /**
     * active() scope
     *
     * @param Builder $query
     * @return void
     */
    public function scopeFeatured(Builder $query): void
    {
        $query->where(fn ($query) => $query->where('featured', 1));
    }

    public function scopeSold(Builder $query): void
    {
        $query->where(fn ($query) => $query->where('is_sold', 1));
    }

    public function scopeNew(Builder $query): void
    {
        $query->where(fn ($query) => $query->where('is_new', 1));
    }

    public function scopeUsed(Builder $query): void
    {
        $query->where(fn ($query) => $query->where('is_new', '!=', 1));
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithViewCount(Builder $query, int $dayRange = 7): void
    {
        $query->addSelect([
            'view_count' => VehicleView::query()
                ->select(DB::raw('SUM(hits)'))
                ->where('date', '>=', Carbon::now()->subDays($dayRange))
                ->whereColumn('vehicle_id', $this->getTable() . '.id')
        ]);
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithSearchViewCount(Builder $query, int $dayRange = 7): void
    {
        $query->addSelect([
            'search_views' => VehicleView::query()
                ->select(DB::raw('SUM(filter_views)'))
                ->where('date', '>=', Carbon::now()->subDays($dayRange))
                ->whereColumn('vehicle_id', $this->getTable() . '.id')
        ]);
    }

    /**
     * @param Builder $query
     * @return void
     */
    public function scopeWithImageCount(Builder $query): void
    {
        $query->addSelect([
            'image_count' => MediaUse::query()
                ->select(DB::raw('COUNT(*)'))
                ->where('owner_type', 'vehicle')
                ->whereColumn('owner_id', $this->getTable() . '.id')
        ]);
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithEnquiryCount(Builder $query, int $dayRange = 7): void
    {
        $query->addSelect([
            'enquiry_count' => Enquiry::query()
                ->select(DB::raw('COUNT(*)'))
                ->where('reason_type', 'vehicle')
                ->whereColumn('reason_id', $this->getTable() . '.id')
        ]);
    }

    public function scopeIsWithinDistance($query, float $lat, float $lng, int $distance = 5)
    {
        $distance_multiplier = Settings::get('automotive-distance_measurement') === 'mi' ? 0.000621371192 : 0.001;
        return $query->whereRaw("ST_Distance_Sphere(point(lng, lat), point(?, ?)) * ? < ?", [
            $lng,
            $lat,
            $distance_multiplier,
            $distance,
        ]);
    }

    /**
     * Sorting - customized from default due to views
     *
     * @param Builder $query
     * @param string|null $sortOption
     * @return Builder
     */
    public function scopeSetSortBy(Builder $query, ?string $sortOption): Builder
    {
        if (empty($sortOption)) {
            return $query;
        }

        if ($sortOption === 'views') {
            return $query->withViewCount()->orderBy('view_count');
        }

        $direction = str_ends_with($sortOption, '_desc') ? 'desc' : 'asc';
        return $query->orderBy(str_replace('_desc', '', $sortOption), $direction);
    }

    /**
     * Find vehicles similar to provided
     *
     * @param Builder $query
     * @param string $matchType
     * @param Vehicle $vehicle
     * @return void
     */
    public function scopeSimilar(Builder $query, string $matchType, self $vehicle): void
    {
        match ($matchType) {
            'price' => $query->orderBy(DB::raw('ABS(price - ' . (int) $vehicle->price . ')')),
            'make' => $query->where('make_id', '=', $vehicle->make_id),
            'price-20' => $query->whereBetween('price', [$vehicle->price * 0.8, $vehicle->price * 1.2]),
            'make-model' => $query->where('make_id', $vehicle->make_id)
                ->where('model_id', $vehicle->model_id),
            'body-type' => $query->where('body_style_id', $vehicle->body_style_id),
            default => $query,
        };
    }

    /**
     * exportable()
     * Stub of active selection for now.
     * Should add settings for making customizing this over time
     *
     * @param Builder $query
     * @return void
     */
    public function scopeExportable(Builder $query): void
    {
        $query->active();
    }

    /**
     * Convert miles to km
     *
     * @param int $miles
     * @return int
     */
    public function milesToKm(int $miles): int
    {
        return (int)($miles * self::KM_IN_MILES);
    }

    /**
     * Convert kilometers to miles
     *
     * @param int $km
     * @return int
     */
    public function kmToMiles(int $km): int
    {
        return (int)($km / self::KM_IN_MILES);
    }

    /**
     * date_in_stock attribute
     * When vehicle was added to stock
     *
     * @return string
     */
    public function getDateInStockAttribute()
    {
        if (!empty($this->attributes['stock_arrival_date'])) {
            return $this->attributes['stock_arrival_date'];
        }

        return $this->created_at;
    }

    /**
     * Condition attribute
     *
     * @return string
     */
    public function getConditionAttribute(): string
    {
        return $this->is_new ? 'new' : 'used';
    }

    /**
     * Status attribute
     *
     * @return string
     */
    public function getStatusAttribute(): string
    {
        if ($this->is_reserved === true) {
            return 'Reserved';
        }

        if ($this->is_sold === true) {
            return 'Sold';
        }

        if (!empty($this->deleted_at)) {
            return 'Archived';
        }

        return $this->is_published ? 'Published' : 'Draft';
    }

    /**
     * Update Status value
     *
     * @param string $status
     * @return void
     */
    public function updateStatus(string $status)
    {
        match ($status) {
            'draft' => $this->update([
                'is_reserved' => false,
                'is_sold' => false,
                'is_published' => false,
            ]),
            'published' => $this->update([
                'is_reserved' => false,
                'is_sold' => false,
                'is_published' => true,
            ]),
            'reserved' => $this->update([
                'is_reserved' => true,
                'is_sold' => false,
                'is_published' => true,
            ]),
            'sold' => $this->update([
                'is_reserved' => false,
                'is_sold' => true,
                'is_published' => false,
            ]),
        };
    }

    public function getDaysInStockAttribute(): int
    {
        if (!empty($this->attributes['stock_arrival_date'])) {
            return Carbon::now()->diffInDays($this->attributes['stock_arrival_date']);
        }

        return Carbon::now()->diffInDays($this->created_at);
    }

    public function getDaysSinceUpdateAttribute(): int
    {
        return Carbon::now()->diffInDays($this->updated_at);
    }

    /**
     * Search name text
     *
     * @return string
     */
    public function getSearchNameAttribute(): string
    {
        return $this->title;
    }

    /**
     * Search excerpt text
     *
     * @return string
     */
    public function getSearchExcerptAttribute(): string
    {
        return collect([
            $this->is_published ? '' : 'DRAFT',
            $this->vrm_condensed,
            $this->asCurrency($this->price ?? 0, Settings::get('app-details-currency')),
            $this->first_registration_date?->format('d/m/Y'),
            $this->derivative,
        ])->filter()->implode(' | ');
    }

    /**
     * Search result icon
     *
     * @return string
     */
    public function getSearchIcon(): string
    {
        return $this->getPreviewImage('mini-thumb')
            ?? 'cars';
    }

    /**
     * Route to viewing a vehicle as a part of search response
     *
     * @return string
     */
    public function getSearchResultRoute(): string
    {
        return UiUrlGenerator::make('manage-content/vehicles/edit/' . $this->id, [], false);
    }

    /**
     * Path to the front-end url
     *
     * @return string
     */
    public function urlPath(): string
    {
        $path = Settings::get('vehicles-url-path');

        $replacements = [
            '{{MAKE}}' => !empty($this->make?->slug) ? $this->make?->slug : 'make',
            '{{MODEL}}' => !empty($this->model?->slug) ? $this->model?->slug : 'model',
            '{{SLUG}}' => $this->slug,
        ];

        return str_replace(array_keys($replacements), $replacements, $path);
    }

    /**
     * Define thumbnail sizes to auto-generate for this model
     *
     * @return mixed
     */
    public function getDefaultAllowedMediaSizesAttribute()
    {
        return Config::get('automotive.vehicle_image_sizes', []);
    }

    public function archive()
    {
        $this->update([
            'is_published' => false,
            'is_archived' => true,
        ]);
    }

    public function getAllCustom()
    {
        $this->loadMissing('attributeValues');

        $custom_values = [];

        $this->attributeValues->each(function ($attribute_value) use (&$custom_values) {
            $custom_values[$attribute_value->slug] = $attribute_value->getValue();
        });

        return $custom_values;
    }

    public function getCustom(string $attribute): mixed
    {
        $value = $this->attributeValues()->where('slug', $attribute)->get()
            ->map(fn(VehicleAttributeValue $value) => $value->getValue());

        if (empty($value) || empty($value->count())) {
            return null;
        }

        return $this->getCustomAttribute($attribute)->count === 1
            ? $value->first()
            : $value;
    }

    public function getMonthlyCostType(): ?string
    {
        if (empty($this->monthly_cost_type)) {
            return null;
        }

        return collect(config('automotive.monthly_cost_types', []))
                ->filter(fn($type) => $type['id'] === $this->monthly_cost_type)
                ->first()['name'] ?? null;
    }

    public function getRegistrationNumberWithoutSpacesAttribute()
    {
        return str_replace(' ', '', $this->registration_number);
    }

    public function getComingSoonStatus(): bool
    {
        return $this->available_date
            ? $this->available_date > Carbon::now()->endOfDay()
            : false;
    }

    /**
     * @param string $attribute
     * @return VehicleAttribute|Model
     */
    protected function getCustomAttribute(string $attribute): VehicleAttribute
    {
        return VehicleAttribute::query()
            ->where('slug', $attribute)
            ->firstOrFail();
    }

    /**
     * @return array
     */
    public function replacementsAttributes(): array
    {
        return [
            'title',
            'derivative',
            'description'
        ];
    }

    public function getStock()
    {
        return 1;
    }
}
