<?php

namespace Mtc\MercuryDataModels;

use Carbon\Carbon;
use Mtc\MercuryDataModels\Factories\InvoiceFactory;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Stancl\Tenancy\Database\Concerns\CentralConnection;

class Invoice extends Model
{
    use HasFactory;
    use CentralConnection;

    public const NO_VAT = 0;
    public const REGULAR_VAT = 20;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'invoices';

    /**
     * Mass assign attributes
     *
     * @var array
     */
    protected $fillable = [
        'tenant_id',
        'email',
        'status',
        'due_at',
        'paid_at',
        'amount',
        'amount_paid',
        'vat_rate',
        'data',
        'pdf_path',
    ];

    /**
     * Cast attributes to types
     *
     * @var array
     */
    protected $casts = [
        'due_at' => 'date',
        'paid_at' => 'datetime',
        'data' => 'array',
    ];

    /**
     * Model Factory
     *
     * @return InvoiceFactory
     */
    protected static function newFactory()
    {
        return InvoiceFactory::new();
    }

    /**
     * Relationship with site
     *
     * @return BelongsTo
     */
    public function tenant(): BelongsTo
    {
        return $this->belongsTo(Tenant::class);
    }

    /**
     * Relationship with invoice items
     *
     * @return HasMany
     */
    public function items(): HasMany
    {
        return $this->hasMany(InvoiceItem::class);
    }

    /**
     * Relationship with invoice addresses
     *
     * @return HasMany
     */
    public function addresses(): HasMany
    {
        return $this->hasMany(InvoiceAddress::class);
    }

    /**
     * Relationship with invoice seller addresses
     *
     * @return HasMany
     */
    public function sellerAddress(): HasOne
    {
        return $this->hasOne(InvoiceAddress::class)->where('type', 'seller');
    }

    /**
     * Relationship with invoice recipient addresses
     *
     * @return HasMany
     */
    public function recipientAddress(): HasOne
    {
        return $this->hasOne(InvoiceAddress::class)->where('type', 'recipient');
    }

    /**
     * Relationship with invoice payments
     *
     * @return HasMany
     */
    public function payments(): HasMany
    {
        return $this->hasMany(InvoicePayment::class);
    }

    /**
     * Relationship with invoice history trace
     *
     * @return HasMany
     */
    public function history(): HasMany
    {
        return $this->hasMany(InvoiceHistory::class);
    }

    public function scopeOverdue(Builder $query, int $days = 30)
    {
        $query->whereNull('paid_at')
            ->where('due_at', '<=', Carbon::now()->subDays($days));
    }

}
