<?php

namespace Mtc\MercuryDataModels\Repositories;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Mtc\MercuryDataModels\Contracts\FeatureFlag;
use Mtc\MercuryDataModels\Tenant;
use Mtc\MercuryDataModels\TenantFeature;

class FeatureRepository
{
    private array $flags = [];

    public function isEnabled(string $feature, bool $forceCheck = false): bool
    {
        if (!isset($this->flags[$feature]) || $forceCheck) {
            $this->flags[$feature] = TenantFeature::query()
                ->where('feature', $feature)
                ->where('tenant_id', tenant('id'))
                ->exists();
        }
        return $this->flags[$feature];
    }

    public function getAll(): Collection
    {
        return collect(config('features.all', []))
            ->filter(fn(string $class) => class_exists($class))
            ->map(fn ($class, $feature) => App::make($class))
            ->filter(fn($class) => $class instanceof FeatureFlag);
    }

    public function enabledForTenant($tenantId = null): Collection
    {
        if (is_null($tenantId)) {
            $tenantId = tenant('id');
        }

        $enabled = TenantFeature::query()
            ->where('tenant_id', $tenantId)
            ->pluck('feature');
        return collect(config('features.all', []))
            ->filter(fn($class, $feature) => class_exists($class) && $enabled->search($feature) !== false)
            ->map(fn ($class, $feature) => App::make($class))
            ->filter(fn($class) => $class instanceof FeatureFlag);
    }

    public function setEnabled(Tenant $tenant, array $features): void
    {
        $tenant->features()
            ->whereNotIn('feature', $features)
            ->delete();

        collect($features)
            ->each(function ($feature) use ($tenant) {
                $tenant->features()->updateOrCreate(['feature' => $feature]);
                if (isset($this->flags[$feature])) {
                    unset($this->flags[$feature]);
                }
            });
    }
}
