<?php

namespace Mtc\MercuryDataModels\Traits;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Mtc\ContentManager\Facades\Fields;
use Mtc\MercuryDataModels\Events\LoadVehicleDataFields;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleAttribute;
use Mtc\MercuryDataModels\VehicleAttributeValue;

trait HasVehicleCustomAttributes
{
    private Collection $allAttributes;

    private function getExtraData(Vehicle $vehicle)
    {
        return collect(event(new LoadVehicleDataFields($vehicle)))
            ->filter()
            ->mapWithKeys(fn($field) => $field)
            ->map(function ($field, $key) {
                $field['readonly'] = true;
                return $field;
            })->values()
            ->merge($this->getCustomAttributes($vehicle));
    }

    private function getCustomAttributes(Vehicle $vehicle): Collection
    {
        return $this->allAttributes()
            ->map(fn(VehicleAttribute $attribute) => $this->allocateAttributeValues($vehicle, $attribute));
    }

    private function allocateAttributeValues(Vehicle $vehicle, VehicleAttribute $attribute): array
    {
        /** @var Collection $values */
        $values = $vehicle->attributeValues
            ->where('attribute_id', $attribute->id)
            ->map(fn(VehicleAttributeValue $value) => [
                'id' => $value->id,
                'value' => $value->getValue(),
            ])->take($attribute->count)
            ->toArray();
        $values = collect($values);
        if ($values->count() < $attribute->count) {
            $values = $values->merge(array_fill(0, $attribute->count - $values->count(), [
                'id' => Str::random(),
                'value' => null,
            ]));
        }
        return [
            'id' => $attribute->id,
            'name' => $attribute->name,
            'slug' => $attribute->slug,
            'type' => $attribute->type,
            'component' => 'EditableContent' . Fields::field($attribute->type)->getComponent(),
            'value' => $values,
            'meta' => $attribute->data,
        ];
    }

    private function allAttributes(): Collection
    {
        if (!isset($this->allAttributes)) {
            $this->allAttributes = VehicleAttribute::all();
        }

        return $this->allAttributes;
    }
}
