<?php

namespace Mtc\MercuryDataModels\Filters;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\MercuryDataModels\Vehicle;

class IsNewFilter extends IsFilter implements CustomPatternFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selections = [])
    {
        $query->where(function ($query) use ($selections) {
            foreach ($selections as $selection) {
                if (strtolower($selection) === 'new') {
                    $query->orWhere('is_new', 1);
                } else {
                    // null or 0
                    $query->orWhere(fn($query) => $query->where('is_new', 0)->orWhereNull('is_new'));
                }
            }
        });
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return collect([
            [
                'id' => 'used',
                'name' => 'Used',
                'count' => null,
                'slug' => 'used',
            ],
            [
                'id' => 'new',
                'name' => 'New',
                'count' => null,
                'slug' => 'new',
            ],
        ]);
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'has_load_more' => $this->hasLoadMore($collection),
            'results' => $collection,
            'hide_in_ui' => true,
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
            'is_range_maximum' => $this->isRangeMaximum(),
            'range_group' => $this->getRangeGroup()
        ];
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return Vehicle::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'Is New';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return '';
    }

    /**
     * Specify attribute on object that represents id
     *
     * @return string
     */
    public function getIdAttribute(bool $for_index = false): string
    {
        return 'is_new';
    }

    /**
     * Specify attribute on object that represents name
     *
     * @return string
     */
    public function getNameAttribute(bool $for_index = false): string
    {
        return 'is_new';
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/is_(new|used)/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        preg_match('/is_(new|used)/', $selection, $matches);
        return $matches[1];
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return $selection
            ? 'is_new'
            : 'is_used';
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return $selection === 'new'
            ? 'New'
            : 'Used';
    }
}
