<?php

namespace Mtc\MercuryDataModels;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Validation\ValidationException;
use Mtc\MercuryDataModels\Factories\BookingFactory;

/**
 * @property ?int $customer_id
 * @property string $registration_number
 * @property ?int $mileage
 * @property ?int $location_id
 * @property ?array $vehicle_data
 * @property string $email
 * @property string $phone_number
 * @property string $first_name
 * @property string $last_name
 * @property Carbon $booking_time
 * @property ?int $delivery_option_id
 * @property string $status
 * @property ?array $data
 */
class Booking extends Model
{
    use HasFactory;

    protected $table = 'service_bookings';

    protected $fillable = [
        'uuid',
        'customer_id',
        'registration_number',
        'mileage',
        'vehicle_data',
        'email',
        'location_id',
        'phone_number',
        'first_name',
        'last_name',
        'booking_time',
        'data',
        'delivery_option_id',
        'status',
        'ingested_at',
    ];

    protected $casts = [
        'booking_time' => 'datetime: Y-m-d H:i',
        'created_at' => 'datetime: Y-m-d H:i:s',
        'vehicle_data' => 'array',
        'data' => 'array',
    ];

    protected $appends = [
        'time',
    ];

    protected static function newFactory(): BookingFactory
    {
        return BookingFactory::new();
    }

    public function packages(): BelongsToMany
    {
        return $this->belongsToMany(
            ServicePackage::class,
            'service_booking_packages',
            'booking_id',
            'package_id'
        );
    }

    public function location(): BelongsTo
    {
        return $this->belongsTo(Dealership::class, 'location_id');
    }

    public function deliveryOption(): BelongsTo
    {
        return $this->belongsTo(BookingDeliveryOption::class, 'delivery_option_id');
    }

    public function setStatus(string $status): void
    {
        if (!in_array($status, array_map(fn($value) => $value->value, BookingStatus::cases()))) {
            throw ValidationException::withMessages(['Invalid Status provided for booking']);
        }

        $this->update([
            'status' => $status
        ]);
    }

    public function getTimeAttribute()
    {
        return $this->booking_time?->format('H:i');
    }
}
