<?php

namespace Mtc\MercuryDataModels;

use Carbon\Carbon;
use Mtc\MercuryDataModels\Factories\ClientInvoiceFactory;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Mtc\MercuryDataModels\Factories\CustomerInvoiceFactory;
use Stancl\Tenancy\Database\Concerns\CentralConnection;

class CustomerInvoice extends Model
{
    use HasFactory;

    public const NO_VAT = 0;
    public const REGULAR_VAT = 20;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'invoices';

    /**
     * Mass assign attributes
     *
     * @var array
     */
    protected $fillable = [
        'preferred_delivery_date',
        'requires_finance',
        'type',
        'status',
        'email',
        'contact_no',
        'status',
        'due_at',
        'paid_at',
        'amount',
        'amount_paid',
        'vat_applicable',
        'data',
        'pdf_path',
    ];

    /**
     * Cast attributes to types
     *
     * @var array
     */
    protected $casts = [
        'due_at' => 'date',
        'paid_at' => 'datetime',
        'data' => 'array',
    ];

    /**
     * Model Factory
     *
     * @return CustomerInvoiceFactory
     */
    protected static function newFactory()
    {
        return CustomerInvoiceFactory::new();
    }

    /**
     * Relationship with invoice items
     *
     * @return HasMany
     */
    public function items(): HasMany
    {
        return $this->hasMany(CustomerInvoiceItem::class);
    }

    /**
     * Relationship with invoice addresses
     *
     * @return HasMany
     */
    public function addresses(): HasMany
    {
        return $this->hasMany(CustomerInvoiceAddress::class);
    }

    /**
     * Relationship with invoice seller addresses
     *
     * @return HasMany
     */
    public function sellerAddress(): HasOne
    {
        return $this->hasOne(CustomerInvoiceAddress::class)->where('type', 'seller');
    }

    /**
     * Relationship with invoice recipient addresses
     *
     * @return HasMany
     */
    public function recipientAddress(): HasOne
    {
        return $this->hasOne(CustomerInvoiceAddress::class)->where('type', 'recipient');
    }

    /**
     * Relationship with invoice payments
     *
     * @return HasMany
     */
    public function payments(): HasMany
    {
        return $this->hasMany(CustomerInvoicePayment::class);
    }

    /**
     * Relationship with invoice history trace
     *
     * @return HasMany
     */
    public function history(): HasMany
    {
        return $this->hasMany(CustomerInvoicePayment::class);
    }

    public function scopeOverdue(Builder $query, int $days = 30)
    {
        $query->whereNull('paid_at')
            ->where('due_at', '<=', Carbon::now()->subDays($days));
    }

}
