<?php

namespace Mtc\MercuryDataModels\Filters;

use Closure;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Mtc\MercuryDataModels\Vehicle;

class EngineSizeMinFilter extends IsFilter
{
    public function applyFilter($query, array $selection = [])
    {
        $query->where('engine_size_cc', '>=', $selection);
    }

    /**
     * Get available results of this filter type
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        $min = floor(Vehicle::query()->where($product_filtering)->min('engine_size_cc'));
        $max = ceil(Vehicle::query()->where($product_filtering)->max('engine_size_cc'));

        $ranges = collect(Config::get('automotive.filter-ranges.engine-size'));
        return $ranges
            // Up to max range including one step over current max
            ->filter(fn ($range, $i) => $range < $max || ($i > 0 && $range > $max && $ranges[$i - 1] < $max))
            ->filter(fn ($range) => $range > $min)
            ->map(fn ($range) => [
                'id' => $range,
                'name' => $this->getSelectionName($range)
            ])
            ->values();
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'has_load_more' => $this->hasLoadMore($collection),
            'results' => $collection,
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
        ];
    }

    public function getModel(): string
    {
        return '';
    }

    public function title(): string
    {
        return 'Minimum Engine Size';
    }

    public function modelSlug(Model $model): string
    {
        return '';
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return round($selection / 1000, 1) . 'L';
    }
}
