<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Mtc\MercuryDataModels\KeyloopLeaseVehicleVariant;
use Mtc\MercuryDataModels\Vehicle;

abstract class IsFilter extends \Mtc\Filter\Contracts\IsFilter
{
    public function __construct(
        protected string $type = '',
        public readonly string $filter_target_model = Vehicle::class
    ) {
        //
    }

    // by default, filter options are hidden if only one option is available
    protected int $min_count_to_show = 2;

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'has_load_more' => $this->hasLoadMore($collection),
            'show' => $this->visibleByDefault(),
            'hide_in_ui' => $this->hideInUi(),
            'min_count_to_show' => $this->minCountToShow(),
            'results' => $collection
                ->filter(fn(Model $model) => !empty(trim($model[$this->getNameAttribute()])))
                ->map(fn(Model $model) => [
                    'id' => $model->slug ?: $model[$this->getIdAttribute()],
                    'name' => $model[$this->getNameAttribute()],
                    'count' => $model->result_count,
                    'slug' => $model->slug
                ]),
        ];
    }

    protected function hideInUi(): bool
    {
        return false;
    }

    protected function minCountToShow()
    {
        return $this->min_count_to_show;
    }

    protected function visibleByDefault(): bool
    {
        return false;
    }

    /**
     * @return bool
     */
    protected function isRangeMaximum(): bool
    {
        return false;
    }

    /**
     * @return string
     */
    protected function getRangeGroup(): string
    {
        return '';
    }

    protected function getQuery(): Builder
    {
        return App::make($this->filter_target_model)->query()
            ->when(
                Settings::get('filter-options-use-active-scope')
                && method_exists($this->filter_target_model, 'scopeActive'),
                fn($query) => $query->active()
            );
    }

    /**
     * @return bool
     */
    protected function filterIsApplicable(): bool
    {
        if ($this->filter_target_model == KeyloopLeaseVehicleVariant::class) {
            return !in_array(
                get_class($this),
                [
                    AgeFilter::class,
                    AgeRangeFilter::class,
                    IsNewFilter::class,
                    IsVatApplicableFilter::class,
                ]
            );
        }

        return true;
    }

    protected function getPriceFilterColumn()
    {
        return $this->filter_target_model == KeyloopLeaseVehicleVariant::class
            ? 'price_ex_vat_ex_vrt'
            : 'price';
    }

    protected function getMonthlyPriceFilterColumn()
    {
        return $this->filter_target_model == KeyloopLeaseVehicleVariant::class
            ? 'cheapest_monthly_price_ex_vat'
            : 'monthly_price';
    }
}
