<?php

namespace Mtc\MercuryDataModels;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class FormActivity extends Model
{
    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'form_activity';

    /**
     * Mass assign attributes
     *
     * @var array
     */
    protected $fillable = [
        'form_id',
        'date',
        'total_submissions',
        'success_count',
        'validation_error_count',
        'failure_count',
    ];

    /**
     * Cast attributes to types
     *
     * @var string[]
     */
    protected $casts = [
        'date' => 'date',
        'total_submissions' => 'integer',
        'success_count' => 'integer',
        'validation_error_count' => 'integer',
        'failure_count' => 'integer',
    ];

    /**
     * Get the form that this activity belongs to
     */
    public function form(): BelongsTo
    {
        return $this->belongsTo(Form::class, 'form_id');
    }

    /**
     * Record a successful form submission
     */
    public static function recordSuccess(string $formId): void
    {
        self::incrementCounter($formId, 'success_count');
    }

    /**
     * Record a validation error (422)
     */
    public static function recordValidationError(string $formId): void
    {
        self::incrementCounter($formId, 'validation_error_count');
    }

    /**
     * Record a failure (400+)
     */
    public static function recordFailure(string $formId): void
    {
        self::incrementCounter($formId, 'failure_count');
    }

    /**
     * Increment a counter for the given form and today's date
     */
    private static function incrementCounter(string $formId, string $column): void
    {
        $today = Carbon::now()->toDateString();

        $activity = self::firstOrCreate(
            ['form_id' => $formId, 'date' => $today],
            [
                'total_submissions' => 0,
                'success_count' => 0,
                'validation_error_count' => 0,
                'failure_count' => 0,
            ]
        );

        $activity->increment('total_submissions');
        $activity->increment($column);
    }
}
