<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\Builder as QueryBuilder;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\MercuryDataModels\Replacement;
use Mtc\MercuryDataModels\Vehicle;

class SearchTermFilter extends IsFilter implements CustomPatternFilter
{
    /**
     * Slug prefix
     */
    protected const TERM_PREFIX = 'q-';

    /**
     * Apply selections to current filtered object
     *
     * @param Builder|QueryBuilder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        if (Settings::get('filter-apply-replacements-to-search-terms')) {
            $selection = $this->applySearchTermReplacements($selection);
        }
        $selection = collect($selection)->flatten()->toArray();
        // Transform possible multiple multi word terms into an array with one word in each
        $short_terms = collect(explode(' ', implode(' ', $selection)))
            ->filter(fn($term) => strlen($term) < 3);
        $searchPhrase = implode(' ', $selection) . '*';

        // Short term is below 3 chars due to mysql full-text index
        $query->where(function ($subQuery) use ($short_terms, $searchPhrase) {
            $subQuery->whereFullText('search_content', $searchPhrase, ['mode' => 'boolean']);
            if ($short_terms->count()) {
                $subQuery->orWhere(function ($q) use ($short_terms) {
                    $short_terms->each(fn($term) => $q->orWhere('search_content', 'like', "%$term%"));
                });
            }
        });

        // Apply default sorting by relevance if no custom sorting is specified
        $query->when(empty(request()->input('sort_by')), function ($query) use ($searchPhrase) {
            $query->orderByRaw("MATCH(search_content) AGAINST(? IN BOOLEAN MODE) DESC", [$searchPhrase]);
        });
    }

    /**
     * Get available results of this filter type
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return collect([]);
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return '';
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return __('filter::filter.labels.search');
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return '';
    }

    /**
     * Specify UI component type used for this filter
     *
     * @return string
     */
    public function uiComponentType(): string
    {
        return '';
    }

    /**
     * Specify whether model should be re-indexed in the index
     * @return bool
     */
    public function reindexModel(): bool
    {
        return false;
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return Str::startsWith($selection, 'q-');
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        return str_replace([self::TERM_PREFIX, '-'], ['', ' '], $selection);
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        if (is_array($selection)) {
            $selection = implode(' ', $selection);
        }
        return self::TERM_PREFIX . Str::slug($selection);
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        if (is_array($selection)) {
            $selection = implode(' ', $selection);
        }
        return __('filter::filter.search_term_name', ['term' => $selection]);
    }

    private function applySearchTermReplacements(array $selection): array
    {
        if (empty($selection)) {
            return $selection;
        }
        $replacements = Replacement::query()
            ->where('type', Vehicle::class)
            ->get();
        foreach ($replacements as $replacement) {
            foreach ($selection as $index => $term) {
                if (stripos($term, $replacement->original) !== false) {
                    $selection[$index] = str_replace(
                        strtolower($replacement->original),
                        strtolower($replacement->replacement),
                        strtolower($term),
                    );
                }
            }
        }
        return $selection;
    }
}
