<?php

namespace Mtc\MercuryDataModels;

use App\Master\InvoiceGenerator;
use App\Tier;
use App\TierHelper;
use Carbon\Carbon;
use Mtc\MercuryDataModels\Factories\TenantFactory;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Stancl\Tenancy\Contracts\TenantWithDatabase;
use Stancl\Tenancy\Database\Concerns\HasDatabase;
use Stancl\Tenancy\Database\Concerns\HasDomains;
use Stancl\Tenancy\Database\Models\Domain;
use Stancl\Tenancy\Database\Models\Tenant as BaseTenant;

/**
 * @property string $id
 * @property boolean $is_suspended
 * @property string $suspended_message
 * @property int $suspended_by
 * @property Carbon $suspended_at
 */
class Tenant extends BaseTenant implements TenantWithDatabase
{
    use HasFactory;
    use HasDomains;
    use HasDatabase;

    private bool $hasMultipleLocations;

    /**
     * @var string[]
     */
    protected $fillable = [
        'name',
        'tier',
        'theme_name',
        'suspended_at',
        'suspended_by',
        'suspended_message',
        'last_invoice_created_at',
        'next_invoice_due_at',
        'currency',
        'port_number',
    ];

    protected $casts = [
        'last_invoice_created_at' => 'datetime',
        'next_invoice_due_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(fn(self $model) => $model->assignPortNumber());
    }

    /**
     * Custom columns that should be saved to db
     *
     * @return string[]
     */
    public static function getCustomColumns(): array
    {
        return [
            'id',
            'name',
            'tier',
            'suspended_at',
            'suspended_by',
            'suspended_message',
            'last_invoice_created_at',
            'next_invoice_due_at',
            'port_number',
        ];
    }

    /**
     * Model factory
     *
     * @return TenantFactory
     */
    protected static function newFactory()
    {
        return TenantFactory::new();
    }

    /**
     * Relationship with users that use this site/tenant
     *
     * @return BelongsToMany
     */
    public function users(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'tenant_users')
            ->withPivot([
                'role',
            ]);
    }

    public function features(): HasMany
    {
        return $this->hasMany(TenantFeature::class);
    }

    /**
     * Primary domain of client
     *
     * @return HasOne
     */
    public function primaryDomain(): HasOne
    {
        return $this->hasOne(config('tenancy.domain_model'), 'tenant_id')->where('primary', 1);
    }

    /**
     * Relationship with user that suspended this tenant
     *
     * @return BelongsTo
     */
    public function suspender(): BelongsTo
    {
        return $this->belongsTo(User::class, 'suspended_by');
    }

    /**
     * Relationship with billing details
     *
     * @return HasOne
     */
    public function billingDetails(): HasOne
    {
        return $this->hasOne(TenantBillingDetail::class, 'tenant_id');
    }

    /**
     * Relationship with billables applied to tenant
     *
     * @return HasMany
     */
    public function billables(): HasMany
    {
        return $this->hasMany(TenantBillable::class);
    }

    /**
     * Relationship with tenant invoices
     *
     * @return HasMany
     */
    public function invoices(): HasMany
    {
        return $this->hasMany(Invoice::class);
    }

    /**
     * is_suspended attribute
     *
     * @return bool
     */
    public function getIsSuspendedAttribute(): bool
    {
        return !empty($this->attributes['suspended_at']);
    }

    /**
     * Scope invoiceDue()
     *
     * @param Builder $query
     * @return void
     */
    public function scopeInvoiceDue(Builder $query)
    {
        $query->whereNotNull('next_invoice_due_at')
            ->where('next_invoice_due_at', '<=', Carbon::now());
    }

    /**
     * Change the tier of a tenant
     *
     * @param string $new_tier
     * @return bool
     */
    public function changeTier(string $new_tier)
    {
        if (TierHelper::isUpgrade($this->tier, $new_tier)) {
            return $this->upgradeToTier($new_tier);
        }

        return $new_tier === $this->tier || $this->downgradeToTier($new_tier);
    }

    /**
     * Check if billing change is required for tenant
     *
     * @return bool
     */
    public function checkBillingChange(): bool
    {
        if (empty($this->change_tier_on_next_invoice_to)) {
            return true;
        }

        $this->switchBillableTierTo($this->change_tier_on_next_invoice_to);
        $this->removeDowngradedFeatures($this->change_tier_on_next_invoice_to);
        $this->tier = $this->change_tier_on_next_invoice_to;
        $this->change_tier_on_next_invoice_to = null;
        return $this->save();
    }

    public function hasMultipleLocations(): bool
    {
        if (!isset($this->hasMultipleLocations)) {
            $this->hasMultipleLocations = Dealership::query()->count() > 1;
        }

        return $this->hasMultipleLocations;
    }

    /**
     * Upgrade tenant to new tier
     *
     * @param string $new_tier
     * @return bool
     */
    private function upgradeToTier(string $new_tier): bool
    {
        $this->switchBillableTierTo($new_tier);
        $this->previous_tier = $this->tier;
        $this->tier = $new_tier;
        (new InvoiceGenerator())->makeForTierUpgrade($this);
        return $this->save();
    }

    /**
     * Downgrade user to tier
     *
     * @param string $new_tier
     * @return bool
     */
    private function downgradeToTier(string $new_tier): bool
    {
        $this->change_tier_on_next_invoice_to = $new_tier;
        return $this->save();
    }

    /**
     * Change the tenants billable tier to $new_tier
     *
     * @param string $new_tier
     * @return void
     */
    private function switchBillableTierTo(string $new_tier)
    {
        $new_tier_billable = Billable::query()->where('code', $new_tier)->firstOrFail();

        $this->billables()
            ->whereHas('billable', fn($query) => $query->where('type', 'tier'))
            ->delete();

        $this->billables()
            ->create([
                'billable_id' => $new_tier_billable->id,
                'price' => $new_tier_billable->price,
                'quantity' => 1,
            ]);
    }

    /**
     * Remove features that customer was using but no longer has access to
     *
     * @param string $new_tier
     * @return void
     */
    private function removeDowngradedFeatures(string $new_tier)
    {
        $this->disableDowngradedSettings($new_tier);
        $this->removeDowngradedBillables($new_tier);
    }

    /**
     * Remove billing records for downgraded features
     *
     * @param string $newTier
     * @return void
     */
    private function disableDowngradedSettings(string $newTier)
    {
        tenancy()->initialize($this->id);

        Setting::query()
            ->whereIn('min_tier', TierHelper::tiersAbove($newTier))
            ->where('type', 'boolean')
            ->where('config_key', 'like', '%.enabled')
            ->update([
                'value' => false,
            ]);

        tenancy()->end();
    }
    /**
     * Remove billing records for downgraded features
     *
     * @param string $new_tier
     * @return void
     */
    private function removeDowngradedBillables(string $new_tier)
    {
        $this->billables()
            ->whereHas(
                'billable',
                fn ($query) => $query->whereIn('type', TierHelper::disallowedFeatures(Tier::from($new_tier)))
            )
            ->delete();
    }

    private function assignPortNumber(): void
    {
        if (empty($this->attributes['port_number'])) {
            $this->port_number = self::query()
                ->max('port_number') + 1;
        }
    }
}
