<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Closure;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\MercuryDataModels\Vehicle;

class EngineSizeMinFilter extends IndexedFilter implements CustomPatternFilter
{
    public static Collection $data;

    public function applyFilter($query, array $selection = [])
    {
        $query->where('engine_size_cc', '>=', $selection);
    }

    /**
     * Get available results of this filter type
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return $this->getData($product_filtering, $limit, $selections)
            ->map(fn ($range) => [
                'id' => $range,
                'value' => $range,
                'name' => $this->getSelectionName($range)
            ])
            ->values();
    }

    protected function getData(Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        if (empty(self::$data)) {
            $max = ceil(Vehicle::query()
                ->when(
                    Settings::get('filter-apply-selections-to-results'),
                    fn($query) => $query->where($product_filtering)
                )
                ->max('engine_size_cc'));

            $ranges = collect(Config::get('automotive.filter-ranges.engine-size'));
            self::$data = $ranges
                // Up to max range including one step over current max
                ->filter(fn ($range, $i) => $range < $max || ($i > 0 && $range > $max && $ranges[$i - 1] < $max));
        }

        return self::$data;
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'hide_in_ui' => $this->hideInUi(),
            'has_load_more' => $this->hasLoadMore($collection),
            'results' => $collection,
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
            'is_range_maximum' => $this->isRangeMaximum(),
            'range_group' => $this->getRangeGroup()
        ];
    }

    public function getModel(): string
    {
        return '';
    }

    public function title(): string
    {
        return 'Engine Size';
    }

    public function modelSlug(Model $model): string
    {
        return '';
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return 'Engine Size From ' . round($selection / 1000, 1) . 'L';
    }

    public function uiComponentType(): string
    {
        return 'range-filter';
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/engine-from-[0-9]+/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        preg_match('/engine-from-([0-9]+)/', $selection, $matches);
        return $matches[1];
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return 'engine-from-' . $selection;
    }

    protected function getRangeGroup(): string
    {
        return 'engine_size';
    }

    public function filterType(): string
    {
        return 'engine_min_size';
    }
}
