<?php

namespace Mtc\MercuryDataModels\Finance\Jobs;

use App\Facades\Settings;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceProvider;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceRequestData;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResultCollection;
use Mtc\MercuryDataModels\Finance\FinanceServiceHelper;
use Mtc\MercuryDataModels\Vehicle;

class FetchVehicleFinanceData implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(
        protected readonly Vehicle $vehicle,
        protected readonly int $term,
        protected readonly int $annual_mileage,
        protected readonly int $deposit,
        protected readonly string $credit_rating,
        protected readonly array $extra,
        protected readonly bool $store_result = true,
    ) {
        $this->onQueue('finance');
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle(): FinanceResultCollection
    {
        $result_collection = $this->fetch($this->prepare());

        return !empty($this->store_result)
            ? $this->store($result_collection)
            : $result_collection;
    }

    public function __get($name)
    {
        if (isset($this->$name)) {
            return $this->$name;
        }
        return null;
    }

    /**
     * Prepare data for fetching
     *
     * @return FinanceRequestData
     */
    protected function prepare(): FinanceRequestData
    {
        return new FinanceRequestData(
            uuid: $this->vehicle->uuid,
            registration_number: $this->vehicle->registration_number,
            cap_id: $this->vehicle->cap_id,
            dealer_id: $this->vehicle->dealership->location_finance ?? '',
            engine_size: $this->vehicle->engine_size_cc,
            condition: $this->vehicle->condition,
            registration_date: $this->vehicle->first_registration_date,
            term: $this->term,
            mileage: $this->vehicle->odometer_mi,
            annual_mileage: $this->annual_mileage,
            price: $this->vehicle->price,
            deposit: $this->deposit,
            credit_rating: $this->credit_rating,
            clientKey: $this->vehicle->dealership?->data[$this->getFinanceProvider()->dealerIdDataField()] ?? null,
            finance_option: $this->vehicle->getCustom('finance_option'),
            extra: $this->getExtraData(),
            is_new: $this->vehicle->is_new ? true : false,
            make_name: $this->vehicle->make?->name,
            is_vat_applicable: $this->vehicle->is_vat_applicable,
        );
    }

    protected function getExtraData(): array
    {
        return array_merge(
            $this->extra ?? [],
            [
                'dealership_data' => $this->vehicle->dealership?->data,
            ],
        );
    }

    /**
     * Fetch data from service
     *
     * @param FinanceRequestData $request
     * @return FinanceResultCollection
     */
    protected function fetch(FinanceRequestData $request): FinanceResultCollection
    {
        $provider = $this->getFinanceProvider();
        return $provider->isEligibleForCreditRatingFinance($this->vehicle)
            ? $provider->calculateRiskBased($request)
            : $provider->calculate($request);
    }

    /**
     * Store result of finance data
     *
     * @param FinanceResultCollection $results
     */
    protected function store(FinanceResultCollection $results): FinanceResultCollection
    {
        if ($results->filter()->count() > 0) {
            // The finance provider found finance examples for this vehicle.
            // Delete previous provider finance examples.
            // Optionally delete previous stock import finance examples.
            $this->vehicle->financeExamples()
                ->when(
                    !Settings::get('finance-overwrite-import-quotes-with-provider-quotes'),
                    fn($query) => $query->whereNot('provider', '=', 'import')
                )
                ->delete();

            $this->vehicle->update([
                'monthly_price' => null,
                'monthly_cost_type' => null,
                'deposit' => null
            ]);

            $results->filter()
                ->each(fn ($result) => $this->vehicle->financeExamples()->create($result->toArray()));

            $lowest_monthly = $results->filter()->sortBy('monthly_price')->first();

            if ($lowest_monthly) {
                $this->vehicle->update([
                    'monthly_price' => $lowest_monthly->monthly_price,
                    'monthly_cost_type' => strtolower($lowest_monthly->finance_type->value),
                    'deposit' => $lowest_monthly->total_deposit
                ]);
            }
        } elseif (Settings::get('finance-delete-examples-when-no-quotes')) {
            // The finance provider didn't find any finance examples.
            // Assume that they no longer offer finance on this vehicle.
            // Delete any old finance provider examples.
            // Keep any finance examples from the stock import.
            $this->vehicle->financeExamples()
                ->whereNot('provider', '=', 'import')
                ->delete();
        }

        return $results;
    }

    /**
     * Gert the finance provider for client
     *
     * @return FinanceProvider
     */
    protected function getFinanceProvider(): FinanceProvider
    {
        return FinanceServiceHelper::initializeForSite();
    }
}
