<?php

namespace Mtc\MercuryDataModels\Jobs;

use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Str;
use Mtc\MercuryDataModels\ApiUsage;

class TrackApiRequest implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * Create a new job instance.
     */
    public function __construct(
        private readonly ?string $tenant_id,
        private readonly ?string $uri,
        private readonly ?array $input,
        private readonly ?string $method,
        private readonly ?int $status_code,
        private readonly ?float $request_length,
    ) {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $usage = ApiUsage::query()->updateOrCreate([
            'tenant_id' => $this->tenant_id,
            'endpoint' => Str::limit($this->uri, 191, ''),
            'method' => $this->method,
            'time_window' => Carbon::now()->format('Y-m-d H')
        ]);
        if ($this->shouldAddSnapshot($usage)) {
            $usage->snapshots()->create([
                'input' => $this->input,
                'request_time' => $this->request_length,
            ]);
        }
        $this->setResponseTimeDetails($usage);
    }

    private function shouldAddSnapshot($usage): bool
    {
        return $this->method === 'POST'
            && !empty($this->input)
            && $this->request_length > config('app.slow_request_time')
            && $usage->snapshots()->count() < 5;
    }

    private function setResponseTimeDetails($usage): void
    {
        $hits = ($usage->hits ?? 0) + 1;
        $avg = $usage->avg + (($this->request_length - $usage->avg) / $hits);

        $usage->update([
            'hits' => $hits,
            'avg' => $avg,
            'min' => ($usage->min ?? 60) > $this->request_length ? $this->request_length : $usage->min,
            'max' => $usage->max < $this->request_length ? $this->request_length : $usage->max,
        ]);
    }
}
