<?php

namespace Mtc\MercuryDataModels\Finance\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\MercuryDataModels\Finance\Config\EvolutionConfig;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceProvider;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceRequestData;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResult;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResultCollection;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceType;
use Mtc\MercuryDataModels\Vehicle;

/**
 * API Docs: https://www.evolutionfunding.com/api/v3/#batch
 */
class EvolutionFinance implements FinanceProvider
{
    public const PCP_LOWEST_AVAILABLE = 101;
    public const PCP_ALL_AVAILABLE = 103;

    public function __construct(
        protected readonly EvolutionConfig $config
    ) {
        //
    }

    /**
     * Perform calculation on Evolution API
     *
     * @param FinanceRequestData $data
     * @return FinanceResultCollection
     */
    public function calculate(FinanceRequestData $data): FinanceResultCollection
    {
        return $this->process(
            $this->call($this->mapRequest($data)),
            $data->deposit,
            $data->annual_mileage,
            $data->price
        );
    }

    public function dealerIdDataField(): string
    {
        return '';
    }

    public function dealerSpecificAccountKey(): string
    {
        return '';
    }

    public function isEligibleForCreditRatingFinance(Vehicle $vehicle): bool
    {
        // not supported in provider
        return false;
    }

    public function calculateRiskBased(FinanceRequestData $data): FinanceResultCollection
    {
        return new FinanceResultCollection();
    }

    /**
     * Map vehicle data to request data
     *
     * @param FinanceRequestData $data
     * @return array
     */
    protected function mapRequest(FinanceRequestData $data): array
    {
            $request = [
                'cprice' => $data->price,
                'cash_deposit' => $data->deposit,
                'cap_code' => $data->cap_id ?? null,
                'annual_miles' => $data->annual_mileage,
                'current_miles' => $data->mileage,
                'reg_date' => $data->registration_date->format('Y-m-d'),
                'term' => $data->term,
                'calc_type' => $this->config->calculationType(),
                'hpnf' => 1,
                'pcp_product' => self::PCP_ALL_AVAILABLE,
            ];

            if ($this->config->hpCalculationValue() > -1) {
                $request['calc_value_hp'] = $this->config->hpCalculationValue();
            }

            if ($this->config->pcpCalculationValue() > -1) {
                $request['calc_value_pcp'] = $this->config->pcpCalculationValue();
            }

            return $request;
    }

    /**
     * Call the api with data
     *
     * @param array $request_data
     * @return array
     */
    protected function call(array $request_data): array
    {
        $response = Http::withHeaders($this->authHeaders())
            ->post($this->endpoint('batch'), $request_data);

        if ($response->failed()) {
            Log::warning('Failed Evolution finance retrieval', [
                'request' => $request_data,
                'client_id' => $this->config->clientId(),
                'status_code' => $response->status(),
                'result' => $response->body(),
            ]);

            return [];
        }
        return $response->json('data');
    }

    /**
     * Process response
     *
     * @param array $response
     * @return FinanceResultCollection
     */
    protected function process(
        array $response,
        float $deposit,
        int $annual_mileage,
        float $price
    ): FinanceResultCollection {
        if (empty($response)) {
            return new FinanceResultCollection();
        }

        $pcp = (new FinanceResultCollection($response['pcp'] ?? []))
            ->flatten(4)
            ->filter(fn($quote) => is_array($quote))
            ->sortBy('regular')
            ->map(fn($quote) => new FinanceResult(
                provider: 'evolution',
                finance_type: FinanceType::PCP,
                monthly_price: $quote['regular'],
                term: $quote['term'],
                number_of_payments: $quote['term'],
                total_deposit: $deposit,
                apr: $quote['apr'],
                first_payment: $quote['first'] ?? null,
                final_payment: $quote['final'] ?? null,
                interest_rate: $quote['rv_interest'] ?? null,
                apply_url: '',
                annual_mileage: $annual_mileage,
                cash_price: $price,
                dealer_deposit: 0,
                customer_deposit: $deposit,
                payable_amount: $quote['total'] ?? 0,
                option_to_purchase_fee: 0,
                documentation_fee: 0,
                product_name: $quote['product_name']
            ))
            ->first();

        $hp = (new FinanceResultCollection($response['hp'] ?? []))
            ->flatten(4)
            ->filter(fn($quote) => is_array($quote))
            ->sortBy('regular')
            ->map(fn($quote) => new FinanceResult(
                provider: 'evolution',
                finance_type: FinanceType::HP,
                monthly_price: $quote['regular'],
                term: $quote['term'],
                number_of_payments: $quote['term'],
                total_deposit: $deposit,
                apr: $quote['apr'],
                first_payment: $quote['first'] ?? null,
                final_payment: $quote['final'] ?? null,
                interest_rate: $quote['rv_interest'] ?? null,
                apply_url: '',
                annual_mileage: $annual_mileage,
                cash_price: $price,
                dealer_deposit: 0,
                customer_deposit: $deposit,
                payable_amount: $quote['total'] ?? 0,
                option_to_purchase_fee: 0,
                documentation_fee: 0,
                product_name: $quote['product_name'],
            ))
            ->first();

        return new FinanceResultCollection([
            $pcp,
            $hp
        ]);
    }

    /**
     * Endpoint for API
     *
     * @param $path
     * @return string
     */
    protected function endpoint($path): string
    {
        return 'https://www.evolutionfunding.com/api/v3/' . $path;
    }

    /**
     * Headers for API
     *
     * @return array
     */
    protected function authHeaders(): array
    {
        return [
            'id' => $this->config->clientId(),
            'password' => $this->config->clientPassword(),
        ];
    }
}
