<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use App\Filter\FilterIndex;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\Traits\FilterIndexIds;
use Mtc\MercuryDataModels\Vehicle;

use function collect;

class BodyTypeFilter extends IndexedFilter
{
    use FilterIndexIds;

    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $sub_types = collect($selection)
            ->filter(fn($type) => Str::startsWith($type, ['small-', 'medium-', 'large-']))
            ->map(fn($type) => [
                'size' => str_replace('-' . str_replace(['small-', 'medium-', 'large-'], '', $type), '', $type),
                'type' => str_replace(['small-', 'medium-', 'large-'], '', $type),
            ]);

        if ($sub_types->isEmpty()) {
            $query->whereIn('body_style_id', $this->getIds($selection, 'body_type'));
            return;
        }

        $base_types = collect($selection)
            ->reject(fn($type) => Str::startsWith($type, ['small-', 'medium-', 'large-']));

        $this->applyFilterWithSubTypes($query, $base_types, $sub_types);
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        $results = Vehicle::query()
            ->distinct()
            ->when(Settings::get('filter-apply-selections-to-results'), fn($query) => $query->where($product_filtering))
            ->pluck('body_style_id');

        $types = BodyStyleType::query()
            ->whereIn('id', $results)
            ->distinct()
            ->get();

        $needs_subtypes = $types->filter(fn(BodyStyleType $type) => $type->has_subcategories);
        $sub_types = collect();
        if ($needs_subtypes->isNotEmpty()) {
            $sub_types = $this->getSubTypes($needs_subtypes);
        }

        $types = $types->reject(fn(BodyStyleType $type) => $type->has_subcategories);
        $sub_types->each(fn($sub_type) => $types->push($sub_type));

        return $types;
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return BodyStyleType::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'Body Types';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->name;
    }

    public function filterType(): string
    {
        return 'body_type';
    }

    private function getSubTypes(Collection $needs_subtypes)
    {
        return $needs_subtypes
            ->map(function (BodyStyleType $type) {
                $small = new BodyStyleType();
                $small->name = 'Small ' . $type->name;
                $small->slug = Str::slug($small->name);

                $medium = new BodyStyleType();
                $medium->name = 'Medium ' . $type->name;
                $medium->slug = Str::slug($medium->name);

                $large = new BodyStyleType();
                $large->name = 'Large ' . $type->name;
                $large->slug = Str::slug($large->name);

                return [
                    $small,
                    $medium,
                    $large,
                ];
            })
            ->flatten(1);
    }

    private function applyFilterWithSubTypes(Builder $query, $base_types, $sub_types)
    {
        $query->where(function ($styleQuery) use ($base_types, $sub_types) {
            $styleQuery->whereIn('body_style_id', $this->getIds($base_types->toArray(), 'body_type'));
            $sub_types->each(function ($sub_type) use ($styleQuery) {
                $body_style = BodyStyleType::query()->whereHas('filterIndex', fn($indexQuery) => $indexQuery
                    ->where('slug', $sub_type['type'])
                    ->where('filter_type', (new BodyStyleType())->getMorphClass()))
                    ->first();
                if ($body_style) {
                    match ($sub_type['size']) {
                        'small' => $styleQuery->orWhere(fn($specificQuery) => $specificQuery
                            ->where('body_style_id', $body_style->id)
                            ->where('vehicle_length', '<=', $body_style->small_if_under)),
                        'large' => $styleQuery->orWhere(fn($specificQuery) => $specificQuery
                            ->where('body_style_id', $body_style->id)
                            ->where('vehicle_length', '>=', $body_style->large_if_above)),
                        'medium' => $styleQuery->orWhere(fn($specificQuery) => $specificQuery
                            ->where('body_style_id', $body_style->id)
                            ->where('vehicle_length', '<=', $body_style->large_if_above)
                            ->where('vehicle_length', '>=', $body_style->small_if_under)),
                        default => $styleQuery->orWhere('body_style_id', $body_style->id),
                    };
                }
            });
        });
    }
}
