<?php

namespace Mtc\MercuryDataModels;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class VehicleStockSyncLog extends Model
{
    public const ACTION_IMPORT = 'import';
    public const ACTION_EXPORT = 'export';

    public const RECORDS_TO_KEEP = 100;
    public const RECORD_KEEP_DAYS = 7;
    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'vehicle_stock_sync_logs';

    protected $fillable = [
        'vehicle_id',
        'provider',
        'action',
        'data',
        'processed',
        'synced_at',
    ];

    protected $casts = [
        'processed' => 'boolean',
        'synced_at' => 'datetime:d/m/Y H:i:s',
        'data' => 'array',
    ];

    public function vehicle(): BelongsTo
    {
        return $this->belongsTo(Vehicle::class);
    }

    /**
     * Scope to get only import logs.
     */
    public function scopeImports(Builder $query): Builder
    {
        return $query->where('action', self::ACTION_IMPORT);
    }

    /**
     * Scope to get only export logs.
     */
    public function scopeExports(Builder $query): Builder
    {
        return $query->where('action', self::ACTION_EXPORT);
    }

    /**
     * Scope to get the last record (by synced_at DESC).
     */
    public function scopeLast(Builder $query): Builder
    {
        return $query->orderByDesc('synced_at')->limit(1);
    }

    public function scopeOld(Builder $query): Builder
    {
        return $query->where('synced_at', '<', now()->subDays(self::RECORD_KEEP_DAYS));
    }
}
