<?php

namespace Mtc\MercuryDataModels\DealBuilder;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Mtc\MercuryDataModels\Customer;
use Mtc\MercuryDataModels\User;
use Mtc\MercuryDataModels\Vehicle;

class Deal extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'customer_id',
        'vehicle_id',
        'status_id',
        'ingested_at',
        'assignee_id',
        'deposit_amount',
        'total_amount',
        'payable_amount',
        'payment_type',
        'contact_number',
        'data',
    ];

    protected $casts = [
        'ingested_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    public function status(): BelongsTo
    {
        return $this->belongsTo(Status::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'assignee_id');
    }

    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    public function vehicle(): BelongsTo
    {
        return $this->belongsTo(Vehicle::class);
    }

    public function addOns(): HasMany
    {
        return $this->hasMany(DealAddOn::class);
    }

    public function finance(): HasMany
    {
        return $this->hasMany(DealFinance::class);
    }

    public function partExchange(): HasMany
    {
        return $this->hasMany(DealPartExchange::class);
    }

    public function history(): HasMany
    {
        return $this->hasMany(DealHistory::class)->latest();
    }

    public function notes(): HasMany
    {
        return $this->hasMany(DealNote::class)->latest();
    }

    public function messages(): HasMany
    {
        return $this->hasMany(DealMessage::class)->latest();
    }

    public function getTimeSinceSubmittedAttribute()
    {
        return $this->timeDiffString($this->ingested_at, Carbon::now());
    }

    public function getTimeSinceUpdatedAttribute()
    {
        return $this->timeDiffString($this->updated_at, Carbon::now());
    }

    protected function timeDiffString(?Carbon $oldest_time, Carbon $youngest_time): string
    {
        if (!$oldest_time) {
            return '';
        }
        $diff = $youngest_time->diff($oldest_time);
        $parts = [];
        if ($diff->d) {
            $parts[] = $diff->d . 'd';
        }
        if ($diff->h) {
            $parts[] = $diff->h . 'h';
        }
        if ($diff->i) {
            $parts[] = $diff->i . 'm';
        }

        return implode(' ', $parts);
    }
}
