<?php

namespace Mtc\MercuryDataModels;

use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Mtc\Crm\Traits\ModelSortAndFilter;
use Mtc\Filter\FilterIndex;
use Mtc\MercuryDataModels\Factories\TransmissionTypeFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Mtc\MercuryDataModels\Traits\EnsuresSlug;
use Mtc\MercuryDataModels\Traits\OrderByName;

class TransmissionType extends Model
{
    use HasFactory;
    use OrderByName;
    use ModelSortAndFilter;
    use EnsuresSlug;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'transmissions';

    /**
     * Mass assign columns
     *
     * @var string[]
     */
    protected $fillable = [
        'id',
        'name',
        'slug',
        'autotrader_id',
    ];

    /**
     * Columns shown in JSON response by default
     *
     * @var string[]
     */
    protected $visible = [
        'id',
        'name',
        'slug',
    ];

    protected static function boot()
    {
        parent::boot();

        self::saving(function (self $transmissionType) {
            $transmissionType->ensureSlug();
        });
    }

    /**
     * Model factory
     *
     * @return TransmissionTypeFactory
     */
    protected static function newFactory()
    {
        return TransmissionTypeFactory::new();
    }

    /**
     * Relationship with vehicles
     *
     * @return HasMany
     */
    public function vehicles(): HasMany
    {
        return $this->hasMany(Vehicle::class, 'transmission_id');
    }

    /**
     * Relationship with offers
     *
     * @return HasMany
     */
    public function offers(): HasMany
    {
        return $this->hasMany(VehicleOffer::class, 'transmission_id');
    }

    /**
     * Relationship with new cars
     *
     * @return HasMany
     */
    public function newCars(): HasMany
    {
        return $this->hasMany(NewCar::class, 'transmission_id');
    }

    public function filterIndex(): MorphOne
    {
        return $this->morphOne(FilterIndex::class, 'filter');
    }

    protected function getSlugAttribute(): string
    {
        if (empty($this->attributes['slug'])) {
            return $this->filterIndex?->slug ?? '';
        }
        return $this->attributes['slug'];
    }
}
