<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\MercuryDataModels\Contracts\MinMaxFilter;

use function collect;

class SeatMinFilter extends IndexedFilter implements CustomPatternFilter, MinMaxFilter
{
    public static Collection $data;

    private ?int $max_value = null;

    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->where('seats', '>=', $selection);
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return $this->getData($product_filtering, $limit, $selections)
            ->filter(fn($range) => $this->isInBounds($range))
            ->map(fn ($seats) => [
                'id' => $seats,
                'name' => $this->getSelectionName($seats),
            ]);
    }

    protected function getData(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        if (empty(self::$data)) {
            self::$data = $this->getQuery()
                ->distinct()
                ->when(
                    Settings::get('filter-apply-selections-to-results'),
                    fn($query) => $query->where($product_filtering)
                )
                ->where('seats', '>', 0)
                ->select('seats')
                ->orderBy('seats')
                ->pluck('seats');
        }
        return self::$data;
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'hide_in_ui' => $this->hideInUi(),
            'has_load_more' => $this->hasLoadMore($collection),
            'results' => $collection,
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
            'is_range_maximum' => $this->isRangeMaximum(),
            'range_group' => $this->getRangeGroup()
        ];
    }

    public function getIndexedResults(string $filter_type, int $limit, array $selections = []): Collection
    {
        return parent::getIndexedResults($filter_type, $limit, $selections)
            ->sortBy(fn($value) => (int)$value->filter_id);
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return $this->filter_target_model;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'Seats';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return '';
    }

    /**
     * Specify attribute on object that represents id
     *
     * @return string
     */
    public function getIdAttribute(bool $for_index = false): string
    {
        return 'seats';
    }

    /**
     * Specify attribute on object that represents name
     *
     * @return string
     */
    public function getNameAttribute(bool $for_index = false): string
    {
        return 'seats';
    }

    public function uiComponentType(): string
    {
        return 'range-filter';
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/from-[0-9]+-seats/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        preg_match('/from-([0-9]+)-seats/', $selection, $matches);
        return $matches[1];
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return 'from-' . $selection . '-seats';
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return $selection === 1
            ? 'From ' . $selection . ' Seat'
            : 'From ' . $selection . ' Seats';
    }

    protected function getRangeGroup(): string
    {
        return 'seats';
    }

    public function filterType(): string
    {
        return 'seats_min';
    }

    public function getOtherBound(): string
    {
        return 'seats_max';
    }

    public function setOtherBoundary(?array $value = null): void
    {
        if (!empty($value)) {
            $this->max_value = $value[0];
        }
    }

    private function isInBounds($value): bool
    {
        if (empty($this->max_value)) {
            return true;
        }
        return $value <= $this->max_value;
    }
}
