<?php

namespace Mtc\MercuryDataModels;

use App\Master\Models\VehicleMake as MasterVehicleMake;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Support\Facades\Storage;
use Mtc\Crm\Traits\ModelSortAndFilter;
use Mtc\Filter\FilterIndex;
use Mtc\MercuryDataModels\Factories\VehicleMakeFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Mtc\MercuryDataModels\Traits\EnsuresSlug;
use Mtc\MercuryDataModels\Traits\OrderByName;

class VehicleMake extends Model
{
    use HasFactory;
    use OrderByName;
    use ModelSortAndFilter;
    use EnsuresSlug;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'vehicle_makes';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'id',
        'slug',
        'name',
        'logo',
        'monochrome_logo',
        'autotrader_id',
        'cap_id',
        'lcv_cap_id',
        'primary_colour',
        'secondary_colour',
        'keyloop_id',
        'keyloop_import_enabled',
        'cars',
        'lcvs',
        'motorcycles',
    ];

    protected $casts = [
        'keyloop_import_enabled' => 'boolean',
    ];

    protected static function boot()
    {
        parent::boot();

        self::saving(function (self $make) {
            $make->ensureSlug();
        });
    }

    /**
     * Relationship with models of the make
     *
     * @return HasMany
     */
    public function models(): HasMany
    {
        return $this->hasMany(VehicleModel::class, 'make_id');
    }

    /**
     * Model factory
     *
     * @return VehicleMakeFactory
     */
    protected static function newFactory()
    {
        return VehicleMakeFactory::new();
    }

    /**
     * Relationship with vehicles
     *
     * @return HasMany
     */
    public function vehicles(): HasMany
    {
        return $this->hasMany(Vehicle::class, 'make_id');
    }

    public function carConfigurations(): HasMany
    {
        return $this->hasMany(CarConfiguratorModel::class, 'make_id');
    }

    public function leaseColours(): HasMany
    {
        return $this->hasMany(KeyloopLeaseColour::class, 'make_id');
    }

    /**
     * Relationship with offers
     *
     * @return HasMany
     */
    public function offers(): HasMany
    {
        return $this->hasMany(VehicleOffer::class, 'make_id');
    }

    /**
     * Relationship with new cars
     *
     * @return HasMany
     */
    public function newCars(): HasMany
    {
        return $this->hasMany(NewCar::class, 'make_id');
    }

    public function filterIndex()
    {
        return $this->morphOne(FilterIndex::class, 'filter');
    }

    protected function logo(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => [
                'path' => $value,
                'url' => !empty($value)
                    ? Storage::disk(config('filesystems.default_media'))->url($value ?? '')
                    : '',
            ],
            set: fn ($value) => is_array($value)
                ? $value['path'] ?? null
                : $value
        );
    }

    protected function monochromeLogo(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => [
                'path' => $value,
                'url' => !empty($value)
                    ? Storage::disk(config('filesystems.default_media'))->url($value ?? '')
                    : '',
            ],
            set: fn ($value) => is_array($value)
                ? $value['path'] ?? null
                : $value
        );
    }
}
