<?php

namespace Mtc\MercuryDataModels\Finance\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\MercuryDataModels\Finance\Config\IVendiConfig;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceProvider;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceRequestData;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResult;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResultCollection;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceType;
use Mtc\MercuryDataModels\Finance\FinanceEligibilityChecker;
use Mtc\MercuryDataModels\Finance\IvendiCreditRating;
use Mtc\MercuryDataModels\Vehicle;

class IVendi implements FinanceProvider
{
    public function __construct(
        protected readonly IVendiConfig $config,
        protected readonly FinanceEligibilityChecker $eligibilityChecker,
    ) {
        //
    }

    public function enabled(): bool
    {
        return $this->config->enabled();
    }

    /**
     * Perform calculation on CodeWeavers API
     *
     * @param FinanceRequestData $data
     * @return FinanceResultCollection
     */
    public function calculate(FinanceRequestData $data): FinanceResultCollection
    {
        return $this->process(
            $this->call($this->mapRequest($data), $data->clientKey),
            $data->deposit
        );
    }

    public function calculateRiskBased(FinanceRequestData $data): FinanceResultCollection
    {
        return $this->process(
            $this->call($this->mapCreditRatingRequest($data), $data->clientKey),
            $data->deposit
        );
    }


    public function dealerIdDataField(): string
    {
        return $this->getDealerIdDataField();
    }

    protected function getDealerIdDataField(): string
    {
        return 'ivendi';
    }

    public function dealerSpecificAccountKey(): string
    {
        return $this->getDealerSpecificAccountKey();
    }

    protected function getDealerSpecificAccountKey(): string
    {
        return 'ivendi-dealer-key';
    }

    public function isEligibleForCreditRatingFinance(Vehicle $vehicle): bool
    {
        return $this->eligibilityChecker->allowCreditRatingFinance($vehicle);
    }

    /**
     * Map request to api data
     *
     * @param FinanceRequestData $data
     * @return array
     */
    protected function mapRequest(FinanceRequestData $data): array
    {
        $identityType = 'VRM';
        $identity = $data->registration_number;
        if (!empty($data->cap_id)) {
            $identity = $data->cap_id;
            $identityType = 'RVC';
        }
        return [
            "Credentials" => [
                "Username" => $this->config->username(),
                "Mode" => 0
            ],
            "DealershipEnrichment" => [
                "UserGenerated" => true
            ],
            "PlatformMeta" => null,
            "ProductRequestUID" => null,
            "IsPlatformProductRequest" => false,
            "QuoteRequests" => [
                "QuoteeUID" => $data->clientKey,
                "GlobalRequestSettings" => null,
                "GlobalRequestParameters" => [
                    "Term" => $data->term
                ],
                "Requests" => [
                    [
                        "CreditTier" => "",
                        "Figures" => [
                            "Asset" => [
                                "AnnualDistance" => $data->annual_mileage,
                                "ManualResidualValue" => null,
                                "VATIncluded" => true,
                                "extras" => null
                            ],
                            "CashDeposit" => $data->deposit,
                            "CashPrice" => $data->price
                        ],
                        "Asset" => [
                            "CurrentOdometerReading" => $data->mileage,
                            "RegistrationDate" => $data->registration_date?->format('d-m-Y'),
                            "RegistrationMark" => $data->registration_number,
                            "Condition" => "Default",
                            "Source" => "Default",
                            "Identity" => $identity,
                            "IdentityType" => $identityType,
                            "StockIdentity" => null,
                            "StockingDate" => null,
                            "StockLocation" => null,
                            "Class" => ucfirst($data->vehicle_type),
                            "CubicCentimetres" => 0,
                            "EntityType" => "None"
                        ],
                        "RequestSettings" => null,
                        "RequestParameters" => null
                    ]
                ]
            ],
        ];
    }

    /**
     * Map request to api data
     *
     * @param FinanceRequestData $data
     * @return array
     */
    protected function mapCreditRatingRequest(FinanceRequestData $data): array
    {
        $registration_date = '';

        if ($data->registration_date) {
            $registration_date = str_replace(' ', 'T', $data->registration_date->format('Y-m-d H:i:s')) . '.000Z';
        }

        return [
            "Credentials" => [
                "Username" => $this->config->username(),
                "Mode" => 0
            ],
            "DealershipEnrichment" => [
                'ClientName' => $this->config->username(),
                'ClientVersion' => '1',
                'IvspDomainId' => 'test-domainUserId',
                'IvspDomainSessionId' => $this->config->ivspDomainSessionId(),
                'IvspNuid' => 'test-networkUserId',
                'UserFingerprint' => 'test-userFingerprint',
                'UserGenerated' => false,
            ],
            "PlatformMeta" => [
                "CashPrice" => $data->price,
                "VehiclePrice" => $data->price,
            ],
            "QuoteRequests" => [
                "QuoteeUID" => $data->clientKey,
                "GlobalRequestParameters" => [
                    "Term" => $data->term
                ],
                "GlobalRequestSettings" => [
                    "AllowAdjustments" => true
                ],
                "Requests" => [
                    [
                        "CreditTier" => $this->getProviderSpecificCreditRating($data->credit_rating),
                        "Figures" => [
                            "Asset" => [
                                "AnnualDistance" => $data->annual_mileage,
                                "ManualResidualValue" => null,
                                "VATIncluded" => true,
                                "extras" => null
                            ],
                            "CashDeposit" => $data->deposit,
                            "CashPrice" => $data->price
                        ],
                        "Asset" => [
                            "CurrentOdometerReading" => $data->mileage,
                            "RegistrationDate" => $registration_date,
                            "RegistrationMark" => $data->registration_number,
                            "Condition" => "used",
                            "Source" => 1,
                            "Identity" => $data->cap_id,
                            "IdentityType" => "RVC",
                            "StockIdentity" => null,
                            "StockingDate" => null,
                            "StockLocation" => null,
                            "Class" => ucfirst($data->vehicle_type),
                            "CubicCentimetres" => null,
                            "EntityType" => "None"
                        ],
                    ]
                ]
            ]
        ];
    }

    /**
     * Call the api with data
     *
     * @param array $request_data
     * @return array
     */
    protected function call(array $request_data, string $specificKey = null): array
    {
        $response = Http::post($this->endpoint('quotes'), $request_data);

        if ($response->failed()) {
            Log::warning('Failed iVendi finance retrieval', [
                'request' => $request_data,
                'api_key' => $specificKey ?? $this->config->apiKey(),
                'status_code' => $response->status(),
                'result' => $response->body(),
            ]);

            return [];
        }

        return $response->json('QuoteResults.0.Results.0.ProductGroups') ?? [];
    }

    /**
     * Process response
     *
     * @param array $response
     * @return FinanceResultCollection
     */
    protected function process(array $response, float $deposit): FinanceResultCollection
    {
        if (empty($response)) {
            return new FinanceResultCollection();
        }

        $product_quotes = [];

        foreach ($response as $product_group) {
            foreach ($product_group['ProductQuotes'] as $product_quote_data) {
                $finance_type = FinanceType::from($product_quote_data['FacilityType']) ?? null;
                $monthly_payment = $product_quote_data['Figures']['RegularPayment'] ?? null;

                if (
                    empty($monthly_payment)
                    || empty($finance_type)
                ) {
                    continue;
                }

                if (!array_key_exists($finance_type->value, $product_quotes)) {
                    $product_quotes[$finance_type->value] = [];
                }

                // get the cheapest quote of each type, ignoring finance option.
                if (
                    empty($product_quotes[$finance_type->value])
                    || $monthly_payment < $product_quotes[$finance_type->value]->monthly_price
                ) {
                    $product_quotes[$finance_type->value] = new FinanceResult(
                        provider: 'ivendi',
                        finance_type: FinanceType::from($product_quote_data['FacilityType']) ?? null,
                        monthly_price: $monthly_payment ?? 0,
                        term: $product_quote_data['Figures']['Term'] ?? 0,
                        number_of_payments: $product_quote_data['Figures']['NumberOfRegularPayments'] ?? 0,
                        total_deposit: $product_quote_data['Figures']['TotalDeposit'] ?? 0,
                        apr: $product_quote_data['Figures']['APR'] ?? 0,
                        first_payment: $product_quote_data['Figures']['FirstPayment'] ?? 0,
                        final_payment: $product_quote_data['Figures']['FinalPayment'] ?? 0,
                        interest_rate: $product_quote_data['Figures']['InterestRate'] ?? 0,
                        apply_url: '',
                        product_name: $product_quote_data['ProductName'] ?? '',
                        annual_mileage: $product_quote_data['Figures']['Asset']['AnnualDistanceQuoted'] ?? 0,
                        cash_price: $product_quote_data['Figures']['TotalCashPrice'] ?? 0,
                        dealer_deposit: $product_quote_data['Figures']['Asset']['FinanceDepositAllowance'] ?? 0,
                        customer_deposit: $deposit,
                        payable_amount: $product_quote_data['Figures']['TotalPayable'] ?? 0,
                        option_to_purchase_fee: $product_quote_data['Figures']['OptionToPurchaseFee'] ?? 0,
                        documentation_fee: 0,
                    );
                }
            }
        }

        return new FinanceResultCollection($product_quotes);
    }

    /**
     * Endpoint for API
     *
     * @param $path
     * @return string
     */
    protected function endpoint($path): string
    {
        return 'https://quoteware3.ivendi.com/' . $path;
    }

    private function getProviderSpecificCreditRating($credit_rating = '')
    {
        return match (strtoupper($credit_rating)) {
            'EXCELLENT' => IvendiCreditRating::EXCELLENT,
            'GOOD' => IvendiCreditRating::GOOD,
            'FAIR' => IvendiCreditRating::FAIR,
            'POOR' => IvendiCreditRating::POOR,
            default => '',
        };
    }
}
