<?php

namespace Mtc\MercuryDataModels\Leasing;

use Mtc\MercuryDataModels\KeyloopLeaseResidualValue;
use Mtc\MercuryDataModels\KeyloopLeaseVehicleVariant;
use Mtc\MercuryDataModels\Leasing\Contracts\LeaseVariantFinance;
use Mtc\MercuryDataModels\Leasing\Contracts\LeaseVariantMonthlyPrice;

abstract class KeyloopLeaseFinanceCalculator
{
    protected array $residual_values = [];
    protected array $discounts = [];

    /**
     * Calculations MUST be assumed to be STRICTLY CONFIDENTIAL.
     * Clients must advise the calculation they wish to use.
     *
     * @param KeyloopFinanceRequest $request
     * @return mixed
     */
    abstract protected function calculateFinance(KeyloopFinanceRequest $request): LeaseVariantMonthlyPrice;

    public function getVariantFinanceOptions(
        KeyloopLeaseVehicleVariant $variant,
        bool $update_cheapest_variant_price = false
    ): ?LeaseVariantFinance {
        $finance_options = new LeaseVariantFinance($variant);
        $cheapest_option = null;

        if (empty($variant->price_ex_vat_ex_vrt)) {
            return null;
        }

        foreach ($this->getContractTerms() as $contract_length_months) {
            foreach ($this->getAnnualMileages() as $annual_mileage) {
                foreach ($this->getDepositMonths() as $deposit_months) {
                    $key = $contract_length_months . '-' . $annual_mileage . '-' . $deposit_months;
                    $finance_option_array = $this->calculateFinance(new KeyloopFinanceRequest(
                        contract_length_months: $contract_length_months ?? 0,
                        deposit_months: $deposit_months ?? 0,
                        item_price_ex_vat_ex_vrt: $variant->price_ex_vat_ex_vrt ?? 0,
                        vrt: $variant->vehicle_vrt ?? 0,
                        discount_percentage_as_decimal: $this->getDiscountPercentage($variant),
                        residual_value_ex_vat: $this->getResidualValueExVat(
                            $variant,
                            $contract_length_months,
                            $annual_mileage
                        ),
                        delivery_price: $variant->delivery_ex_vat_ex_vrt ?? 0,
                        bonus_value: $this->getBonusValue($variant),
                        electric_grant: $this->getElectricGrant($variant),
                    ));
                    $finance_options->monthly_prices[$key] = $finance_option_array;

                    if (
                        empty($cheapest_option)
                        || $finance_option_array->monthly_price_ex_vat < $cheapest_option->monthly_price_ex_vat
                    ) {
                        $cheapest_option = $finance_option_array;
                    }
                }
            }
        }

        if ($cheapest_option && $update_cheapest_variant_price) {
            $variant->update([
                'cheapest_monthly_price_ex_vat' => $cheapest_option->monthly_price_ex_vat,
                'cheapest_monthly_price_inc_vat' => $cheapest_option->monthly_price_inc_vat,
            ]);
        }

        $finance_options->cheapest_monthly_price = $cheapest_option;

        return $finance_options;
    }

    /**
     * @param KeyloopLeaseVehicleVariant $variant
     * @param array $item
     * @return LeaseVariantFinance
     */
    public function getItemFinanceOptions(KeyloopLeaseVehicleVariant $variant, array $item): LeaseVariantFinance
    {
        $finance_options = new LeaseVariantFinance($variant);
        $cheapest_option = null;

        foreach ($this->getContractTerms() as $contract_length_months) {
            foreach ($this->getAnnualMileages() as $annual_mileage) {
                foreach ($this->getDepositMonths() as $deposit_months) {
                    // financeable items may take discount percentage linked to the vehicle variant
                    $finance_option_array = $this->calculateFinance(new KeyloopFinanceRequest(
                        contract_length_months: $contract_length_months,
                        deposit_months: $deposit_months,
                        item_price_ex_vat_ex_vrt: $item['price_ex_vat_ex_vrt'] ?? 0,
                        vrt: $item['vrt'] ?? 0,
                        discount_percentage_as_decimal: $this->getDiscountPercentage($variant),
                        vat_as_decimal: $item['vat'] ?? null,
                    ));

                    $key = $contract_length_months . '-' . $annual_mileage . '-' . $deposit_months;
                    $finance_options->monthly_prices[$key] = $finance_option_array;

                    if (
                        empty($cheapest_option)
                        || $finance_option_array->monthly_price_ex_vat < $cheapest_option->monthly_price_ex_vat
                    ) {
                        $cheapest_option = $finance_option_array;
                    }
                }
            }
        }

        $finance_options->cheapest_monthly_price = $cheapest_option;

        return $finance_options;
    }

    /**
     * Calculate possible monthly road tax values for each available contract combination.
     *
     * @param KeyloopLeaseVehicleVariant $variant
     * @return LeaseVariantFinance
     */
    public function getRoadTaxFinanceOptions(KeyloopLeaseVehicleVariant $variant)
    {
        $finance_options = new LeaseVariantFinance($variant);
        $cheapest_option = null;

        foreach ($this->getContractTerms() as $contract_length_months) {
            $contract_length_years = $contract_length_months / 12;
            $contract_road_tax = $variant->road_fund_licence * $contract_length_years;

            foreach ($this->getAnnualMileages() as $annual_mileage) {
                foreach ($this->getDepositMonths() as $deposit_months) {
                    // financeable items may take discount percentage linked to the vehicle variant
                    $finance_option_array = $this->calculateFinance(new KeyloopFinanceRequest(
                        contract_length_months: $contract_length_months,
                        deposit_months: $deposit_months,
                        item_price_ex_vat_ex_vrt: $contract_road_tax,
                        vat_as_decimal: 0,
                    ));

                    $key = $contract_length_months . '-' . $annual_mileage . '-' . $deposit_months;
                    $finance_options->monthly_prices[$key] = $finance_option_array;

                    if (
                        empty($cheapest_option)
                        || $finance_option_array->monthly_price_ex_vat < $cheapest_option->monthly_price_ex_vat
                    ) {
                        $cheapest_option = $finance_option_array;
                    }
                }
            }
        }

        $finance_options->cheapest_monthly_price = $cheapest_option;

        return $finance_options;
    }

    protected function getResidualValueExVat(
        KeyloopLeaseVehicleVariant $variant,
        int $contract_length_months,
        int $annual_mileage
    ): float {
        return $this->getVariantResidualValue($variant, $contract_length_months, $annual_mileage) ?? 0;
    }

    protected function getDiscountPercentage(KeyloopLeaseVehicleVariant $variant): float
    {
        return $this->getVariantDiscounts($variant, 'percentage') ?? 0;
    }

    protected function getBonusValue(KeyloopLeaseVehicleVariant $variant): float
    {
        return $this->getVariantDiscounts($variant, 'cash') ?? 0;
    }

    protected function getElectricGrant(KeyloopLeaseVehicleVariant $variant): float
    {
        // electric grant is not currently available in data source
        return 0;
    }

    protected function getContractTerms()
    {
        return config('leasing.keyloop.contract_terms');
    }

    protected function getAnnualMileages()
    {
        return config('leasing.keyloop.annual_mileages');
    }

    protected function getDepositMonths()
    {
        return config('leasing.keyloop.deposit_quantity_months');
    }

    protected function getVariantDiscounts(KeyloopLeaseVehicleVariant $variant, $type): float
    {
        // Note that we assume there will only be one discount of each type
        if (empty($this->discounts)) {
            $this->discounts = [
                'percentage' => ($variant->getDiscounts('percentage')->first()['discount_amount'] ?? 0) / 100.0,
                'cash' => $variant->getDiscounts('cash')->first()['discount_amount'] ?? 0,
            ];
        }

        return $this->discounts[$type] ?? 0;
    }

    protected function getVariantResidualValue(
        KeyloopLeaseVehicleVariant $variant,
        int $contract_length_months,
        int $annual_mileage
    ): float {
        $separator = '_';
        if (empty($this->residual_values)) {
            $this->residual_values = KeyloopLeaseResidualValue::query()
                ->where('external_variant_id', $variant->external_variant_id)
                ->get()
                ->mapWithKeys(function ($residual_value) use ($separator) {

                    $key = $residual_value->contract_length
                        . $separator . $residual_value->annual_mileage;

                    return [
                        $key => $residual_value->residual_value_excluding_vat,
                    ];
                })->toArray();
        }

        return $this->residual_values[$contract_length_months . $separator . $annual_mileage] ?? 0;
    }
}
