<?php

namespace Mtc\MercuryDataModels\Jobs;

use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Mail;
use Mtc\MercuryDataModels\ApiNotification;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleView;

class VehicleComparisonMailRequest implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * Create a new job instance.
     */
    public function __construct(
        private readonly string $email,
        private readonly array $vehicle_slugs,
        private readonly string $comparison_url,
    ) {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $success = false;
        $message = '';

        try {
            $this->trackComparisonRequests();
            $this->sendMail();
            $success = true;
            $message = 'success';
        } catch (\Exception $exception) {
            $message = get_class($exception) . ': ' . $exception->getMessage();
        } finally {
            ApiNotification::query()->create([
                'provider' => 'comparison_email_request',
                'processed' => $success,
                'data' => [
                    'slugs' => $this->vehicle_slugs,
                    'recipient' => $this->email,
                    'url' => $this->comparison_url,
                    'message' => $message,
                ],
            ]);
        }
    }

    private function trackComparisonRequests(): void
    {
        $vehicle_ids = collect($this->vehicle_slugs)->map(fn ($slug) =>
            Vehicle::query()->where('slug', $slug)->first()?->id ?? null)
            ->filter();

        $existing = VehicleView::query()
            ->whereIn('vehicle_id', $vehicle_ids->toArray())
            ->where('date', Carbon::now()->format('Y-m-d'))
            ->pluck('vehicle_id');

        $missing = $vehicle_ids->reject(fn($id) => $existing->contains($id));

        if ($missing->isNotEmpty()) {
            $data = $missing->map(fn($id) => [
                'vehicle_id' => $id,
                'compare_email_requests' => 1,
                'date' => Carbon::now()->format('Y-m-d')
            ]);
            VehicleView::query()->upsert($data->toArray(), ['vehicle_id', 'date']);
        }

        VehicleView::query()
            ->whereIn('vehicle_id', $existing)
            ->increment('compare_email_requests');
    }

    private function sendMail(): void
    {
        $class = config('crm.vehicle_comparison_email');
        Mail::to($this->email)->send(new $class($this->comparison_url));
    }
}
