<?php

namespace Mtc\MercuryDataModels\Services;

use App\Facades\Settings;
use Exception;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Support\Facades\Log;
use Mtc\MercuryDataModels\Finance\Contracts\BatchRequestsQuotes;
use Mtc\MercuryDataModels\Finance\Contracts\ExportsStockToFinance;
use Mtc\MercuryDataModels\Finance\FinanceServiceHelper;
use Mtc\MercuryDataModels\Finance\Jobs\FetchCapBasedVehicleFinanceData;
use Mtc\MercuryDataModels\Finance\Jobs\FetchVehicleFinanceData;
use Mtc\MercuryDataModels\Finance\Jobs\RunFinanceExport;
use Mtc\MercuryDataModels\Finance\Jobs\RunQuoteRequest;
use Mtc\MercuryDataModels\Vehicle;

class FinanceService
{
    use DispatchesJobs;

    public function request(Vehicle $vehicle, array $data = [], ?bool $store = null): bool
    {
        try {
            $this->dispatch(new FetchVehicleFinanceData(
                $vehicle,
                $data['term'] ?? Settings::get('finance-term'),
                $data['annual_mileage'] ?? Settings::get('finance-annual_mileage'),
                $this->getDepositValue($data['deposit'] ?? null, $vehicle),
                $data['credit_rating'] ?? Settings::get('finance-credit_rating'),
                $data,
                // Explicitly set by request, if not set then check setting
                $store ?? Settings::get('finance-store-user-requested-quotes') ?? false
            ));
            return true;
        } catch (Exception $exception) {
            Log::error($exception->getMessage());
        }
        return false;
    }

    public function requestAllVehicles(): bool
    {
        try {
            if (!FinanceServiceHelper::hasEnabledProvider()) {
                return false;
            }

            $provider = FinanceServiceHelper::initializeForSite();
            if ($provider instanceof BatchRequestsQuotes && Settings::get('finance-batch-update')) {
                $this->dispatch(new RunQuoteRequest($provider));
                return true;
            }
        } catch (Exception $exception) {
            Log::error($exception->getMessage());
        }
        return false;
    }

    public function get(Vehicle $vehicle, array $data = []): array
    {
        try {
            $fetchRequest = new FetchVehicleFinanceData(
                $vehicle,
                $data['term'] ?? Settings::get('finance-term'),
                $data['annual_mileage'] ?? Settings::get('finance-annual_mileage'),
                $this->getDepositValue($data['deposit'] ?? null, $vehicle),
                $data['credit_rating'] ?? Settings::get('finance-credit_rating'),
                $data,
                Settings::get('finance-store-user-requested-quotes') ?? false
            );
            return $fetchRequest->handle()->toArray();
        } catch (Exception $exception) {
            Log::error($exception->getMessage());
        }
        return $this->errorMessage();
    }


    public function getForCapId(
        string $cap_id,
        int $price,
        ?int $term = null,
        ?int $deposit = null,
        ?int $annual_mileage = null,
        ?string $credit_rating = null,
        ?string $product_type = null
    ): array {
        try {
            $fetchRequest = new FetchCapBasedVehicleFinanceData(
                $cap_id,
                $price,
                $term ?? Settings::get('finance-term'),
                $annual_mileage ?? Settings::get('finance-annual_mileage'),
                $this->getDepositValue($deposit ?? null, new Vehicle(['price' => $price])),
                $credit_rating ?? Settings::get('finance-credit_rating'),
                ['type' => $product_type],
                Settings::get('finance-store-user-requested-quotes') ?? false
            );
            return $fetchRequest->handle()->toArray();
        } catch (Exception $exception) {
            Log::error($exception->getMessage());
        }
        return $this->errorMessage();
    }

    public function triggerExport(): bool
    {
        try {
            $provider = FinanceServiceHelper::initializeForSite();
            if ($provider instanceof ExportsStockToFinance && Settings::get('finance-export-to-provider')) {
                Log::info('Finance export triggered for ' . tenant('id') . ' via controller');
                $this->dispatch(new RunFinanceExport($provider));
                return true;
            }
        } catch (Exception $exception) {
            Log::error($exception->getMessage());
        }
        return false;
    }

    private function errorMessage(): array
    {
        return [
            'status' => 'fail',
            'message' => 'Could not request finance calculation for the request',
        ];
    }

    private function getDepositValue(?int $deposit, Vehicle $vehicle): float
    {
        // User selected deposit
        if (!is_null($deposit) && $deposit >= 0) {
            if (Settings::get('finance-deposit-sub-100-deposit-percentage')) {
                // Deposit above 100% is treated as deposit amount
                return ($deposit > 100)
                    ? $deposit
                    : $deposit * 0.01 * $vehicle->price;
            }

            return $deposit;
        }

        $deposit_to_return = Settings::get('finance-deposit_percentage') * 0.01 * $vehicle->price;

        // check if vehicle price requires flat rate deposit
        if (
            Settings::get('finance-deposit-flat-rate-treshold') > 0
            && $vehicle->price < Settings::get('finance-deposit-flat-rate-treshold')
        ) {
            $deposit_to_return = Settings::get('finance-deposit-flat-rate-amount');
        }

        // ensure deposit >= minimum deposit
        if (
            Settings::get('finance-minimum-deposit-amount') > 0
            && $deposit_to_return < Settings::get('finance-minimum-deposit-amount')
        ) {
            $deposit_to_return = Settings::get('finance-minimum-deposit-amount');
        }

        return $deposit_to_return;
    }
}
