<?php

namespace Mtc\MercuryDataModels\Filters;

use Illuminate\Support\Collection;
use Closure;
use Mtc\MercuryDataModels\Contracts\MinMaxFilter;

class PriceMonthlyMaxFilter extends PriceMonthlyMinFilter implements MinMaxFilter
{
    private ?int $min_value = null;

    public function applyFilter($query, array $selection = [])
    {
        $query->where($this->getMonthlyPriceFilterColumn(), '<=', $selection);
    }

    public function getResults(Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return $this->getData($product_filtering, $limit, $selections)
            ->filter(fn($range) => $this->isInBounds($range))
            ->map(fn($range) => [
                'id' => $range,
                'value' => $range,
                'name' => $this->getSelectionName($range),
            ])
            ->values();
    }


    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/price-monthly-up-to-[0-9]+/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        preg_match('/price-monthly-up-to-([0-9]+)/', $selection, $matches);
        return $matches[1];
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return 'Price Up To ' . $this->asCurrency((float)$selection);
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return 'price-monthly-up-to-' . $selection;
    }

    protected function isRangeMaximum(): bool
    {
        return true;
    }

    public function filterType(): string
    {
        return 'price_monthly_max';
    }

    public function getOtherBound(): string
    {
        return 'price_monthly_min';
    }

    public function setOtherBoundary(?array $value = null): void
    {
        if (!empty($value)) {
            $this->min_value = $value[0];
        }
    }

    private function isInBounds($value): bool
    {
        if (empty($this->min_value)) {
            return true;
        }
        return $value >= $this->min_value;
    }
}
