<?php

namespace Mtc\MercuryDataModels;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Mtc\ContentManager\Traits\HasMedia;
use Mtc\ContentManager\Traits\ModelSortAndFilter;
use Mtc\MercuryDataModels\Contracts\ModelWithUrlPath;
use Mtc\MercuryDataModels\Factories\CarConfiguratorModelFactory;
use Mtc\MercuryDataModels\Traits\EnsuresSlug;

class CarConfiguratorModel extends Model implements ModelWithUrlPath
{
    use SoftDeletes;
    use HasFactory;
    use HasMedia;
    use EnsuresSlug;
    use ModelSortAndFilter;

    public static $relationship_models = [
        'editions',
        'trims',
        'engines',
        'colours',
        'wheels',
        'interiors',
        'packages',
        'extras',
        'custom',
    ];

    protected $table = 'car_configurator_models';

    protected $fillable = [
        'name',
        'slug',
        'make_id',
        'model_id',
        'active',
        'data',
    ];

    protected $casts = [
        'active' => 'bool',
        'data' => 'array',
    ];

    protected static function newFactory()
    {
        return CarConfiguratorModelFactory::new();
    }

    protected static function boot()
    {
        parent::boot();

        self::creating(fn(self $entry) => $entry->ensureSlug());
    }

    public function make(): BelongsTo
    {
        return $this->belongsTo(VehicleMake::class);
    }

    public function model(): BelongsTo
    {
        return $this->belongsTo(VehicleModel::class);
    }

    public function sections(): HasMany
    {
        return $this->hasMany(CarConfiguratorSection::class, 'model_id')->orderBy('order');
    }

    public function activeSections(): HasMany
    {
        return $this->sections()->where('active', 1);
    }

    public function custom(): HasMany
    {
        return $this->hasMany(CarConfiguratorCustomSection::class, 'model_id')
            ->orderBy('order');
    }

    public function trims(): HasMany
    {
        return $this->hasMany(CarConfiguratorTrim::class, 'model_id')
            ->orderBy('order');
    }

    public function editions(): HasMany
    {
        return $this->hasMany(CarConfiguratorEdition::class, 'model_id')
            ->orderBy('order');
    }

    public function engines(): HasMany
    {
        return $this->hasMany(CarConfiguratorEngine::class, 'model_id')
            ->orderBy('order');
    }

    public function colours(): HasMany
    {
        return $this->hasMany(CarConfiguratorColour::class, 'model_id')
            ->orderBy('order');
    }

    public function wheels(): HasMany
    {
        return $this->hasMany(CarConfiguratorWheel::class, 'model_id')
            ->orderBy('order');
    }

    public function interiors(): HasMany
    {
        return $this->hasMany(CarConfiguratorInterior::class, 'model_id')
            ->orderBy('order');
    }

    public function packages(): BelongsToMany
    {
        return $this->belongsToMany(
            CarConfiguratorPackage::class,
            'car_configurator_model_packages',
            'model_id',
            'package_id'
        )
            ->withPivot(['price', 'order']);
    }

    public function extras(): BelongsToMany
    {
        return $this->belongsToMany(
            CarConfiguratorExtra::class,
            'car_configurator_model_extras',
            'model_id',
            'extra_id'
        )
            ->withPivot(['price', 'order']);
    }

    public function restrictions(): HasMany
    {
        return $this->hasMany(CarConfiguratorRestriction::class, 'model_id');
    }

    public function scopeSetSelections(Builder $query, array $selections = [])
    {
        collect($selections)
            ->each(function ($selection) use ($query) {
                match ($selection['type']) {
                    'status' => $this->oneOfStatuses($query, $selection['values']),
                    'make_id',
                    'model_id' => $query->when(
                        in_array(null, $selection['values']),
                        fn($subQuery) => $subQuery->where(fn($nullableQuery) => $nullableQuery
                            ->whereNull($selection['type'])
                            ->orWhereIn($selection['type'], $selection['values'])),
                        fn($subQuery) => $subQuery->whereIn($selection['type'], $selection['values'])
                    ),
                    default => $query
                };
            });
    }

    public function oneOfStatuses(Builder $query, $selections): void
    {
        $query->where(function ($statusQuery) use ($selections) {
            foreach ($selections as $selection) {
                match ($selection) {
                    'active' => $statusQuery->orWhere($selection, 1),
                    'draft' => $statusQuery->orWhere('active', 0),
                    default => $statusQuery,
                };
            }
        });
    }

    public function urlPath(): string
    {
        $path = Settings::get('configurator-url-path');

        $replacements = [
            '{{MAKE}}' => !empty($this->make?->slug) ? $this->make?->slug : 'make',
            '{{MODEL}}' => !empty($this->model?->slug) ? $this->model?->slug : 'model',
            '{{SLUG}}' => $this->slug,
        ];

        return str_replace(array_keys($replacements), $replacements, $path);
    }
}
