<?php

namespace Mtc\MercuryDataModels;

use App\Facades\Settings;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Mtc\ContentManager\Traits\HasMedia;
use Mtc\MercuryDataModels\Contracts\ModelWithContent;
use Mtc\MercuryDataModels\Contracts\ModelWithUrlPath;
use Mtc\MercuryDataModels\Contracts\SearchableModel;
use Mtc\MercuryDataModels\Factories\DealershipFactory;
use Mtc\MercuryDataModels\Services\LocatingService;
use Mtc\MercuryDataModels\Tools\UiUrlGenerator;
use Mtc\MercuryDataModels\Traits\EnsuresSlug;
use Mtc\MercuryDataModels\Traits\OrderByName;
use OwenIt\Auditing\Contracts\Auditable;

use function config;

class Dealership extends Model implements SearchableModel, Auditable, ModelWithUrlPath, ModelWithContent
{
    use \OwenIt\Auditing\Auditable;
    use HasFactory;
    use HasMedia;
    use OrderByName;
    use EnsuresSlug;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'dealerships';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'name',
        'email',
        'active',
        'contact_no',
        'address1',
        'address2',
        'city',
        'county',
        'postcode',
        'country',
        'open_times',
        'coordinates',
        'lat',
        'lng',
        'contact_form_id',
        'location_finance',
        'location_stock',
        'data',
        'timezone',
        'franchise_id',
        'notes',
        'template_id',
        'alt_open_times',
        'second_alt_open_times',
        'external_showroom_url',
    ];

    /**
     * Cast attributes to types
     *
     * @var string[]
     */
    protected $casts = [
        'open_times' => 'array',
        'alt_open_times' => 'array',
        'second_alt_open_times' => 'array',
        'data' => 'array',
        'active' => 'boolean',
    ];

    /**
     * Model factory
     *
     * @return DealershipFactory
     */
    protected static function newFactory()
    {
        return DealershipFactory::new();
    }

    protected static function boot()
    {
        parent::boot();

        self::creating(fn(self $dealership) => $dealership->setDefaultOpenTimes());
        self::retrieved(fn(self $dealership) => $dealership->setDefaultOpenTimes());
        self::saving(function (self $dealership) {
            $dealership->ensureSlug();
            $dealership->setCoordinateFields();
        });
    }

    public function getOwnerColumn(): string
    {
        return 'dealership_id';
    }

    /**
     * @return BelongsTo
     */
    public function form(): BelongsTo
    {
        return $this->belongsTo(config('crm.form_model'));
    }

    public function serviceAvailability(): HasOne
    {
        return $this->hasOne(BookingAvailability::class, 'dealership_id');
    }

    public function serviceBookings(): HasMany
    {
        return $this->hasMany(Booking::class, 'location_id');
    }

    /**
     * Relationship with vehicles that are located in this dealership
     *
     * @return HasMany
     */
    public function vehicles(): HasMany
    {
        return $this->hasMany(Vehicle::class, 'dealership_id');
    }

    public function offers(): HasMany
    {
        return $this->hasMany(VehicleOffer::class, 'dealership_id');
    }

    public function allHolidays(): HasMany
    {
        return $this->hasMany(DealershipHoliday::class, 'dealership_id');
    }

    public function upcomingHolidays(): HasMany
    {
        return $this->hasMany(DealershipHoliday::class, 'dealership_id')
            ->where('date', '>=', Carbon::now());
    }

    /**
     * Relationship with content of the dealership
     *
     * @return HasMany
     */
    public function content(): HasMany
    {
        return $this->hasMany(DealershipContent::class, 'dealership_id')
            ->whereNull('parent_id')
            ->orderBy('order');
    }

    /**
     * Relationship with content of the dealership
     *
     * @return HasMany
     */
    public function allContent(): HasMany
    {
        return $this->hasMany(DealershipContent::class, 'dealership_id')
            ->orderBy('order');
    }

    /**
     * Dealership content history versions
     *
     * @return HasMany
     */
    public function versions(): HasMany
    {
        return $this->hasMany(DealershipContentHistory::class, 'dealership_id')->latest();
    }

    /**
     * Relationship with template that defines content
     *
     * @return BelongsTo
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.template_model'));
    }

    public function scopeNearPostcode(Builder $query, ?string $postcode)
    {
        if (empty($postcode)) {
            return;
        }
        try {
            $service = App::make(LocatingService::class)->locate($postcode);
            $query->orderByRaw("ST_Distance_Sphere(point(lng, lat), point(?, ?)) ", [
                $service->lng(),
                $service->lat(),
            ]);
        } catch (Exception) {
            // Ignore when not found or issue
        }
    }

    /**
     * @param Builder $query
     * @return void
     */
    public function scopeWithVehicleCount(Builder $query): void
    {
        $query->addSelect([
            'vehicle_count' => Vehicle::query()
                ->select(DB::raw('COUNT(*)'))
                ->whereColumn('dealership_id', $this->getTable() . '.id')
        ]);
    }

    /**
     * Scope a query to search dealerships based on a term.
     *
     * @param Builder $query
     * @param string|null $term
     * @return Builder
     */
    public function scopeSearch(Builder $query, ?string $term): Builder
    {
        if (!$term) {
            return $query;
        }

        $searchTerm = '%' . $term . '%';

        return $query->where(function ($query) use ($searchTerm) {
            $query->where('name', 'like', $searchTerm)
                ->orWhere('address1', 'like', $searchTerm)
                ->orWhere('address2', 'like', $searchTerm)
                ->orWhere('city', 'like', $searchTerm)
                ->orWhere('county', 'like', $searchTerm)
                ->orWhere('postcode', 'like', $searchTerm);
        });
    }

    /**
     * is_open attribute
     *
     * @return bool|null
     */
    public function getIsOpenAttribute(): ?bool
    {
        $now = Carbon::now($this->getDealershipTimeZone());
        $openTime = $this->getOpeningTimes()
            ->where('dayOfWeek', $now->dayOfWeek)
            ->first() ?? [];
        if (empty($openTime)) {
            return null;
        }

        return $openTime['open'] <= $now->format('H:i')
            && $openTime['close'] >= $now->format('H:i');
    }

    /**
     * closes_at attribute
     *
     * @return null|string
     */
    public function getClosesAtAttribute(): ?string
    {
        if ($this->getIsOpenAttribute() === false) {
            return null;
        }

        $now = Carbon::now($this->getDealershipTimeZone());
        $openTime = $this->getOpeningTimes()
            ->where('dayOfWeek', $now->dayOfWeek)
            ->first();

        return $openTime['close'] ?? '';
    }

    /**
     * opens_at attribute
     *
     * @return null|string
     */
    public function getOpensAtAttribute(): null|array
    {
        if ($this->getIsOpenAttribute() === true) {
            return null;
        }
        $daysInFuture = 0;
        $openTime = [];

        $now = Carbon::now($this->getDealershipTimeZone());
        $openTimes = $this->getOpeningTimes();
        // Find the closest day in future when location will be open
        do {
            $daysInFuture++;
            $weekDayToCheck = $now->dayOfWeek + $daysInFuture;
            if ($weekDayToCheck > 7) {
                $weekDayToCheck -= 7;
            }

            if ($openTimes->where('dayOfWeek', $weekDayToCheck)->isNotEmpty()) {
                $openTime = [
                    'day' => $weekDayToCheck,
                    'time' => $openTimes->where('dayOfWeek', $weekDayToCheck)->first(),
                ];
            }
            if ($daysInFuture >= 7) {
                break;
            }
        } while (empty($openTime));

        return $openTime['time'] ?? null;
    }

    /**
     * Search name text
     *
     * @return string
     */
    public function getSearchNameAttribute(): string
    {
        return $this->name;
    }

    /**
     * Search excerpt text
     *
     * @return string
     */
    public function getSearchExcerptAttribute(): string
    {
        return '';
    }

    /**
     * Define thumbnail sizes to auto-generate for this model
     *
     * @return mixed
     */
    public function getDefaultAllowedMediaSizesAttribute()
    {
        return Config::get('automotive.dealership_image_sizes', []);
    }

    /**
     * Search result icon
     *
     * @return string
     */
    public function getSearchIcon(): string
    {
        return 'garage-car';
    }

    /**
     * Route to viewing a vehicle as a part of search response
     *
     * @return string
     */
    public function getSearchResultRoute(): string
    {
        return UiUrlGenerator::make('manage-content/dealerships/edit/' . $this->id, [], false);
    }

    public function allocateHolidays(array $holidays)
    {
        collect($holidays)
            ->each(fn($holiday) => $this->upcomingHolidays()->updateOrCreate(['date' => $holiday]));
    }

    public function urlPath(): string
    {
        return '/dealership/' . $this->slug;
    }

    public function departments(): HasMany
    {
        return $this->hasMany(DealershipDepartment::class, 'dealership_id');
    }

    public function primaryDepartment(): HasOne
    {
        return $this->hasOne(DealershipDepartment::class, 'dealership_id')->where('is_primary', '=', true);
    }

    public function franchise()
    {
        return $this->belongsTo(Franchise::class, 'franchise_id');
    }

    private function ensureSlug()
    {
        if (empty($this->attributes['slug'])) {
            $this->slug = Str::slug($this->name);
        }
    }

    private function setDefaultOpenTimes(): void
    {
        if ($this->primaryDepartment) {
            $this->open_times = $this->primaryDepartment->open_times;
            return;
        }

        if (is_array($this->open_times) && !empty($this->open_times)) {
            // use dealerships.open_times
            return;
        }

        $opens = Settings::get('automotive-dealership-open-time');
        $closes = Settings::get('automotive-dealership-close-time');

        $this->open_times = collect(range(1, 7))
            ->map(fn($dayOfWeek) => [
                'dayOfWeek' => $dayOfWeek,
                'open' => $this->isOpenOnDayOfWeek($dayOfWeek) ? $opens : null,
                'close' => $this->isOpenOnDayOfWeek($dayOfWeek) ? $closes : null,
            ])->toArray();
    }

    private function isOpenOnDayOfWeek(int $dayOfWeek): bool
    {
        return match (Settings::get('automotive-dealership-open-days')) {
            'workdays' => $dayOfWeek <= 5,
            'except-sunday' => $dayOfWeek != 7,
            'all' => true,
            default => false,
        };
    }

    private function getDealershipTimeZone(): ?string
    {
        return Settings::get('app-dealership-timezone') && $this->timezone
            ? $this->timezone
            : Settings::get('app-timezone');
    }

    private function setCoordinateFields()
    {
        if (!empty($this->attributes['coordinates'])) {
            $coords = explode(',', $this->attributes['coordinates']);
            $this->lat = $coords[0] ?? null;
            $this->lng = $coords[1] ?? null;
        }
    }

    private function getOpeningTimes(): Collection
    {
        return collect(
            $this->primaryDepartment
                ? $this->primaryDepartment->open_times
                : json_decode($this->attributes['open_times'], true)
        );
    }
}
