<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Mtc\MercuryDataModels\Traits\FilterIndexIds;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;

class MakeFilter extends IndexedFilter
{
    use FilterIndexIds;

    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->whereIn('make_id', $this->getIds($selection, 'make'));
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        $modified_query = '';

        $query = Vehicle::query()
            ->distinct()
            ->when(Settings::get('filter-apply-selections-to-results'), function ($query) use ($product_filtering, &$modified_query) {
                // remove the SELECT from the query, as we only want the WHERE clause.
                $modified_query = str_ireplace(
                    'select distinct * from `vehicles` where',
                    '',
                    $query->where($product_filtering)->toSql());

                if (Settings::get('automotive-vehicle-filters-filter-makes-by-selected-model') === false) {
                    /*
                     * Modify the query to exclude the model restriction.
                     * If the query contains WHERE `model_id` IN (1, 2, 3)
                     * we change it to WHERE -1 NOT IN (1, 2, 3).
                     * This should allow all makes to be returned regardless of selected model.
                     */
                    $modified_query = str_ireplace('`model_id` IN', '-1 NOT IN', $modified_query);
                }
            });

        $results = DB::table('vehicles')->whereRaw($modified_query, $query->getBindings())
            ->pluck('make_id');

        return VehicleMake::query()
            ->with('filterIndex')
            ->whereIn('id', $results)
            ->orderBy('name')
            ->distinct()
            ->get();
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return VehicleMake::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'Make';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->name;
    }

    /**
     * Whether this
     * @return bool
     */
    public function visibleByDefault(): bool
    {
        return true;
    }

    public function filterType(): string
    {
        return 'make';
    }
}
