<?php

namespace Mtc\MercuryDataModels;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;
use Mtc\MercuryDataModels\Factories\PropertyCategoryFactory;
use Mtc\MercuryDataModels\Factories\TransmissionTypeFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Mtc\MercuryDataModels\Traits\EnsuresSlug;
use Mtc\MercuryDataModels\Traits\OrderByName;

class PropertyCategory extends Model
{
    use HasFactory;
    use OrderByName;
    use EnsuresSlug;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'property_categories';

    /**
     * Mass assign columns
     *
     * @var string[]
     */
    protected $fillable = [
        'name',
        'slug',
        'active',
        'data',
    ];

    protected $casts = [
        'active' => 'boolean',
        'data' => 'array',
    ];

    protected static function boot()
    {
        parent::boot();
        self::creating(static fn(self $category) => $category->ensureSlug());
    }

    /**
     * Model factory
     *
     * @return PropertyCategoryFactory
     */
    protected static function newFactory()
    {
        return PropertyCategoryFactory::new();
    }

    public function properties(): HasMany
    {
        return $this->hasMany(Property::class, 'category_id');
    }

    public function fields(): HasMany
    {
        return $this->hasMany(PropertyCategoryField::class, 'category_id');
    }

    public function scopeActive(Builder $query): void
    {
        $query->where('active', 1);
    }

    public function scopeWithActivePropertyCount(Builder $query): void
    {
        $query->addSelect([
            'count_active' => Property::query()
                ->selectRaw('COUNT(*)')
                ->whereColumn('category_id', $this->getTable() . '.id')
                ->where('active', 1)
        ]);
    }
}
