<?php

namespace Mtc\MercuryDataModels;

use App\Facades\Settings;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Mtc\ContentManager\Contracts\MediaUse;
use Mtc\ContentManager\Models\Template;
use Mtc\ContentManager\Traits\HasMedia;
use Mtc\Crm\Traits\ModelSortAndFilter;
use Mtc\MercuryDataModels\Contracts\ModelWithContent;
use Mtc\MercuryDataModels\Contracts\ModelWithUrlPath;
use Mtc\MercuryDataModels\Contracts\SearchableModel;
use Mtc\MercuryDataModels\Factories\VehicleOfferFactory;
use Mtc\MercuryDataModels\Tools\UiUrlGenerator;
use Mtc\MercuryDataModels\Traits\EnsuresSlug;
use OwenIt\Auditing\Contracts\Auditable;

/**
 * @property ?Template $template
 * @property ?VehicleOfferContent $content
 */
class VehicleOffer extends Model implements SearchableModel, ModelWithUrlPath, Auditable, ModelWithContent
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;
    use HasMedia;
    use ModelSortAndFilter;
    use EnsuresSlug;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'vehicle_offers';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'uuid',
        'stock_provider',
        'vin',
        'type_id',
        'published',
        'featured',
        'name',
        'derivative',
        'dealership_id',
        'template_id',
        'make_id',
        'model_id',
        'form_id',
        'price',
        'deposit',
        'seo',
        'data',
        'published_at',
        'engine_size_cc',
        'colour',
        'battery_range',
        'battery_capacity_kwh',
        'battery_usable_capacity_kwh',
        'battery_charge_time',
        'battery_quick_charge_time',
        'franchise_id',
        'fuel_type_id',
        'drivetrain_id',
        'body_style_id',
        'transmission_id',
    ];

    /**
     * Cast attributes to types
     *
     * @var string[]
     */
    protected $casts = [
        'published_at' => 'datetime',
        'published' => 'boolean',
        'featured' => 'boolean',
        'seo' => 'array',
        'data' => 'array',
    ];

    protected $appends = [
        'status',
    ];

    /**
     * Model Factory
     *
     * @return VehicleOfferFactory
     */
    protected static function newFactory()
    {
        return VehicleOfferFactory::new();
    }

    protected static function boot()
    {
        parent::boot();
        self::creating(function (self $offer) {
            $offer->ensureSlug();
            $offer->setDefaultValues();
        });
    }

    /**
     * Relationship with model
     *
     * @return BelongsTo
     */
    public function make(): BelongsTo
    {
        return $this->belongsTo(VehicleMake::class, 'make_id');
    }

    /**
     * Relationship with model
     *
     * @return BelongsTo
     */
    public function model(): BelongsTo
    {
        return $this->belongsTo(VehicleModel::class, 'model_id');
    }

    /**
     * Relationship with content of the offer
     *
     * @return HasMany
     */
    public function content(): HasMany
    {
        return $this->hasMany(VehicleOfferContent::class, 'offer_id')
            ->whereNull('parent_id')
            ->orderBy('order');
    }

    /**
     * Relationship with content of the offer
     *
     * @return HasMany
     */
    public function allContent(): HasMany
    {
        return $this->hasMany(VehicleOfferContent::class, 'offer_id')
            ->orderBy('order');
    }

    /**
     * Relationship with drivetrain
     *
     * @return BelongsTo
     */
    public function drivetrain(): BelongsTo
    {
        return $this->belongsTo(DrivetrainType::class, 'drivetrain_id');
    }

    /**
     * Relationship with fuel type
     *
     * @return BelongsTo
     */
    public function fuelType(): BelongsTo
    {
        return $this->belongsTo(FuelType::class, 'fuel_type_id');
    }

    /**
     * Relationship with transmission
     *
     * @return BelongsTo
     */
    public function transmission(): BelongsTo
    {
        return $this->belongsTo(TransmissionType::class, 'transmission_id');
    }

    /**
     * Relationship with bodyStyle
     *
     * @return BelongsTo
     */
    public function bodyStyle(): BelongsTo
    {
        return $this->belongsTo(BodyStyleType::class, 'body_style_id');
    }

    /**
     * Offer content history versions
     *
     * @return HasMany
     */
    public function versions(): HasMany
    {
        return $this->hasMany(VehicleOfferContentHistory::class, 'offer_id')->latest();
    }

    /**
     * Relationship with dealership
     *
     * @return BelongsTo
     */
    public function dealership(): BelongsTo
    {
        return $this->belongsTo(Dealership::class);
    }

    /**
     * Relationship with franchise
     *
     * @return BelongsTo
     */
    public function franchise(): BelongsTo
    {
        return $this->belongsTo(Franchise::class);
    }

    /**
     * Relationship with finance data for offer
     *
     * @return HasMany
     */
    public function finance(): HasMany
    {
        return $this->hasMany(VehicleOfferFinance::class, 'offer_id');
    }

    /**
     * Relationship with template that defines content
     *
     * @return BelongsTo
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.template_model'));
    }

    /**
     * Relationship with offer type
     *
     * @return BelongsTo
     */
    public function offerType(): BelongsTo
    {
        return $this->belongsTo(OfferType::class, 'type_id');
    }

    /**
     * Relationship with specs
     *
     * @return MorphMany
     */
    public function specs(): MorphMany
    {
        return $this->morphMany(VehicleTechnicalData::class, 'vehicle');
    }

    public function features(): MorphMany
    {
        return $this->morphMany(VehicleFeature::class, 'vehicle');
    }

    public function equipment(): MorphMany
    {
        return $this->morphMany(VehicleStandardEquipment::class, 'vehicle');
    }

    public function form(): BelongsTo
    {
        return $this->belongsTo(Form::class);
    }

    public function labels(): BelongsToMany
    {
        return $this->belongsToMany(Label::class, 'vehicle_offer_labels');
    }

    /**
     * Relationship with offer views on a particular date
     *
     * @return HasMany
     */
    public function views(): HasMany
    {
        return $this->hasMany(OfferView::class, 'offer_id');
    }

    public function conversions(): MorphMany
    {
        return $this->morphMany(Conversion::class, 'owner');
    }

    /**
     * All enquiries submitted for this offer
     *
     * @return MorphMany
     */
    public function allEnquiries(): MorphMany
    {
        return $this->morphMany(config('crm.enquiry_model'), 'reason');
    }

    /**
     * Enquiries submitted for this offer in current month
     *
     * @return MorphMany
     */
    public function currentMonthEnquiries(): MorphMany
    {
        return $this->morphMany(config('crm.enquiry_model'), 'reason')
            ->where('created_at', '>=', Carbon::now()->startOfMonth());
    }

    /**
     * active() scope
     *
     * @param Builder $query
     * @return void
     */
    public function scopeActive(Builder $query)
    {
        $query->where('published', 1)
            ->where(fn ($query) => $query
                ->whereNull('published_at')
                ->orWhere('published_at', '<=', Carbon::now())
            );
    }

    public function scopeWithListingRelationships(Builder $query)
    {
        $query
            ->with([
                'primaryMediaUse.media',
                'make.filterIndex',
                'model.filterIndex',
                'fuelType',
                'offerType',
                'transmission',
                'drivetrain',
                'bodyStyle',
            ])
            ->when(Settings::get('offer-list-load-features'), fn($query) => $query->with('features'));
    }


    public function scopeWhereModelSlug(Builder $query, ?string $slug): void
    {
        if ($slug) {
            $query->whereHas('model.filterIndex', fn ($indexQuery) => $indexQuery->where('slug', $slug));
        }
    }

    /**
     * Find vehicles similar to provided
     *
     * @param Builder $query
     * @param string $matchType
     * @param Vehicle $vehicle
     * @return void
     */
    public function scopeSimilar(Builder $query, string $matchType, self $vehicle)
    {
        match ($matchType) {
            'price' => $query->orderBy(DB::raw('ABS(price - ' . (int) $vehicle->price . ')')),
            'make' => $query->orderBy(DB::raw('make_id = - ' . (int) $vehicle->make_id)),
            default => $query,
        };
    }

    /**
     * New vehicle type filter
     *
     * @param Builder $query
     * @return void
     */
    public function scopeNew(Builder $query): void
    {
        $query->where('type', 'new');
    }

    /**
     * Motability type filter
     *
     * @param Builder $query
     * @return void
     */
    public function scopeMotability(Builder $query): void
    {
        $query->where('type', 'motability');
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithViewCount(Builder $query, int $dayRange = 7): void
    {
        $query->addSelect([
            'view_count' => OfferView::query()
                ->selectRaw('SUM(hits)')
                ->where('date', '>=', Carbon::now()->subDays($dayRange))
                ->whereColumn('offer_id', $this->getTable() . '.id')
        ]);
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithSearchViewCount(Builder $query, int $dayRange = 7): void
    {
        $query->addSelect([
            'search_views' => OfferView::query()
                ->selectRaw('SUM(filter_views)')
                ->where('date', '>=', Carbon::now()->subDays($dayRange))
                ->whereColumn('offer_id', $this->getTable() . '.id')
        ]);
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithImageCount(Builder $query, int $dayRange = 7): void
    {
        $query->addSelect([
            'image_count' => MediaUse::query()
                ->selectRaw('COUNT(*)')
                ->where('owner_type', 'offer')
                ->whereColumn('owner_id', $this->getTable() . '.id')
        ]);
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithEnquiryCount(Builder $query, int $dayRange = 7): void
    {
        $query->addSelect([
            'enquiry_count' => Enquiry::query()
                ->selectRaw('COUNT(*)')
                ->where('reason_type', 'offer')
                ->whereColumn('reason_id', $this->getTable() . '.id')
        ]);
    }


    /**
     * Sorting - customized from default due to views
     *
     * @param Builder $query
     * @param string|null $sortOption
     * @return Builder
     */
    public function scopeSetSortBy(Builder $query, ?string $sortOption): Builder
    {
        if (empty($sortOption)) {
            return $query;
        }

        if ($sortOption === 'views') {
            return $query->withViewCount()->orderBy('view_count');
        }

        $direction = str_ends_with($sortOption, '_desc') ? 'desc' : 'asc';
        return $query->orderBy(str_replace('_desc', '', $sortOption), $direction);
    }

    /**
     * Search name text
     *
     * @return string
     */
    public function getSearchNameAttribute(): string
    {
        return $this->name;
    }

    /**
     * Search excerpt text
     *
     * @return string
     */
    public function getSearchExcerptAttribute(): string
    {
        return strip_tags($this->description);
    }

    public function getDaysInStockAttribute(): int
    {
        return Carbon::now()->diffInDays($this->created_at);
    }

    public function getDaysSinceUpdateAttribute(): int
    {
        return Carbon::now()->diffInDays($this->updated_at);
    }

    /**
     * Search result icon
     *
     * @return string
     */
    public function getSearchIcon(): string
    {
        return 'badge-dollar';
    }

    /**
     * Route to viewing a vehicle as a part of search response
     *
     * @return string
     */
    public function getSearchResultRoute(): string
    {
        return UiUrlGenerator::make('manage-content/offers/edit/' . $this->id, [], false);
    }

    /**
     * Status attribute
     *
     * @return string
     */
    public function getStatusAttribute(): string
    {
        return $this->published ? 'Published' : 'Draft';
    }

    /**
     * Path to the front-end url
     *
     * @return string
     */
    public function urlPath(): string
    {
        return '/offer/' . $this->slug;
    }

    /**
     * Define thumbnail sizes to auto-generate for this model
     *
     * @return mixed
     */
    public function getDefaultAllowedMediaSizesAttribute()
    {
        return Config::get('automotive.offer_image_sizes', []);
    }

    protected function setDefaultValues()
    {
        if (empty($this->attributes['template_id'])) {
            $this->template_id = Template::query()->where('slug', 'offer-show')->first()?->id;
        }

        if (empty($this->attributes['form_id'])) {
            $this->form_id = Form::query()
                ->whereHas('type', fn($query) => $query->where('name', 'Offer Enquiry'))
                ->first()?->id;
        }
    }

    /**
     * Set filters based on terms passed
     *
     * @param Builder $query
     * @param array|string $filters
     * @return void
     */
    public function scopeSetFilters(Builder $query, array|string $filters): void
    {
        collect($filters)
            ->filter()
            ->each(function ($filter) use ($query) {
                if (str_starts_with($filter, 'franchise-')) {
                    $query->whereHas('franchise', fn($query) => $query->where('slug', str_replace('franchise-', '', $filter)));
                } elseif (method_exists($this, 'scope' . ucfirst(Str::camel($filter)))) {
                    $scopeMethod = 'scope' . ucfirst(Str::camel($filter));
                    $this->{$scopeMethod}($query);
                } else {
                    $query->whereHas('offerType', fn ($typeQuery) => $typeQuery->where('slug', $filter));
                }
            });
    }

    /**
     * @return array
     */
    public function replacementsAttributes(): array
    {
        return [
            'name',
        ];
    }
}
