<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use App\Filter\FilterIndex;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Colour;
use Mtc\MercuryDataModels\SearchIndex;
use Mtc\MercuryDataModels\Vehicle;

use function collect;

class ColourFilter extends IndexedFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $colourNameQuery = FilterIndex::select('name')
            ->where('filter_type', '=', 'colour')
            ->whereIn('slug', $selection);

        $query->when(
            Settings::get('filter-use-colour-mapping'),
            fn ($query) => $query->whereHas('colourMap.colour', fn ($colourQuery) =>
                $colourQuery->whereIn('colour', $colourNameQuery)),
            fn ($query) => $query->whereIn('colour', $colourNameQuery),
        );
    }

    /**
     * Get available results of this filter type
     *
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        if (Settings::get('filter-use-colour-mapping')) {
            return Colour::query()
                ->whereHas('mappedColours.vehicles', fn($query) => $query
                    ->when(Settings::get('filter-apply-selections-to-results'), fn($query) => $query->where($product_filtering))
                    ->where(fn ($query) => $product_filtering($query)))
                ->distinct()
                ->get();
        }

        return Vehicle::query()
            ->when(Settings::get('filter-apply-selections-to-results'), fn($query) => $query->where($product_filtering))
            ->select('colour')
            ->distinct()
            ->get();
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        $settings = app()->has('site-settings');
        if ($settings && Settings::get('filter-use-colour-mapping')) {
            return Colour::class;
        }
        return Vehicle::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'Colour';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->colour ?? '';
    }

    /**
     * Specify attribute on object that represents id
     *
     * @return string
     */
    public function getIdAttribute(bool $for_index = false): string
    {
        return 'colour';
    }

    /**
     * Specify attribute on object that represents name
     *
     * @return string
     */
    public function getNameAttribute(bool $for_index = false): string
    {
        return 'colour';
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'hide_in_ui' => $this->hideInUi(),
            'has_load_more' => $this->hasLoadMore($collection),
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
            'results' => $this->formatResults($collection)
        ];
    }

    public function filterType(): string
    {
        return 'colour';
    }

    private function formatResults(Collection $collection): Collection
    {
        if ($collection->first() instanceof SearchIndex) {
            return $collection->map(fn($colour) => [
                'id' => $colour->slug,
                'name' => $colour->name,
                'count' => $colour->num_results,
                'slug' => $colour->slug,
            ]);
        }

        $colour_indexes = FilterIndex::where('filter_type', '=', 'colour')
            ->whereIn('name', $collection->pluck('colour'))
            ->get()
            ->mapWithKeys(function ($index) {
                return [$index['name'] => $index['slug']];
            });

        return $collection
            ->filter(fn(Model $model) => !empty(trim($model[$this->getNameAttribute()])))
            ->filter(fn(Model $model) => $colour_indexes->has($model[$this->getNameAttribute()]))
            ->map(function (Model $model) use ($colour_indexes) {
                return [
                    'id' => $colour_indexes[$model[$this->getNameAttribute()]] ?? $model[$this->getIdAttribute()],
                    'name' => $model[$this->getNameAttribute()],
                    'count' => $model->result_count,
                    'slug' => $colour_indexes[$model[$this->getNameAttribute()]],
                ];
            });
    }
}
