<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use App\Traits\FormatAsCurrency;
use Closure;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\MercuryDataModels\Vehicle;

class PriceMonthlyMinFilter extends IndexedFilter implements CustomPatternFilter
{
    use FormatAsCurrency;

    public static Collection $data;

    public function applyFilter($query, array $selection = [])
    {
        $query->where('monthly_price', '>=', $selection);
    }

    public function getResults(Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return $this->getData($product_filtering, $limit, $selections)
            ->map(fn($range) => [
                'id' => $range,
                'value' => $range,
                'name' => $this->getSelectionName($range),
            ])
            ->slice(0, -1)
            ->values();
    }

    protected function getData(Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        if (empty(self::$data)) {
            $min = floor(Vehicle::query()
                ->when(
                    Settings::get('filter-apply-selections-to-results'),
                    fn($query) => $query->where($product_filtering)
                )
                ->min('monthly_price'));
            $max = ceil(Vehicle::query()
                ->when(
                    Settings::get('filter-apply-selections-to-results'),
                    fn($query) => $query->where($product_filtering)
                )
                ->max('monthly_price'));

            $ranges = collect(Config::get('automotive.filter-ranges.monthly_price'));
            self::$data = $ranges
                // Up to max range including one step over current max
                ->filter(fn($range, $i) => $range < $max || ($i > 0 && $range > $max && $ranges[$i - 1] < $max))
                ->filter(fn($range, $i) => $range > $min
                    || ($i < $ranges->count() - 1 && $range < $min && $ranges[$i + 1] > $min));
        }
        return self::$data;
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'hide_in_ui' => $this->hideInUi(),
            'has_load_more' => $this->hasLoadMore($collection),
            'results' => $collection,
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
            'is_range_maximum' => $this->isRangeMaximum(),
            'range_group' => $this->getRangeGroup()
        ];
    }

    public function getIndexedResults(string $filter_type, int $limit, array $selections = []): Collection
    {
        return parent::getIndexedResults($filter_type, $limit, $selections)
            ->sortBy(fn($value) => (int)$value->filter_id);
    }

    public function getModel(): string
    {
        return '';
    }

    public function title(): string
    {
        return 'Monthly Price';
    }

    public function modelSlug(Model $model): string
    {
        return '';
    }

    public function uiComponentType(): string
    {
        return 'range-filter';
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/price-monthly-from-[0-9]+/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        preg_match('/price-monthly-from-([0-9]+)/', $selection, $matches);
        return $matches[1];
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return 'Price From ' . $this->asCurrency($selection);
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return 'price-monthly-from-' . $selection;
    }

    protected function getRangeGroup(): string
    {
        return 'monthly_price';
    }

    public function filterType(): string
    {
        return 'price_monthly_min';
    }
}
