<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Traits\FilterIndexIds;
use Mtc\MercuryDataModels\VehicleMake;

class MakeFilter extends IndexedFilter
{
    use FilterIndexIds;

    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->whereHas('make', fn($makeQuery) => $makeQuery->whereIn('slug', $selection));
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        $results = $this->getQuery()
            ->distinct()
            ->when(Settings::get('filter-apply-selections-to-results'), fn($query) => $query->where($product_filtering))
            ->pluck('make_id');

        return VehicleMake::query()
            ->whereIn('id', $results)
            ->orderBy('name')
            ->distinct()
            ->get();
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return VehicleMake::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'Make';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->name;
    }

    /**
     * Whether this
     * @return bool
     */
    public function visibleByDefault(): bool
    {
        return true;
    }

    public function filterType(): string
    {
        return 'make';
    }
}
