<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\MercuryDataModels\Vehicle;

class ManufactureYearMinFilter extends IndexedFilter implements CustomPatternFilter
{
    public static Collection $data;

    public function applyFilter($query, array $selection = [])
    {
        $query->where('manufacture_year', '>=', $selection);
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return $this->getData($product_filtering, $limit, $selections)
            ->map(fn ($year) => [
                'id' => $year,
                'name' => $this->getSelectionName($year),
            ]);
    }

    protected function getData(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        if (empty(self::$data)) {
            self::$data = Vehicle::query()
                ->when(
                    Settings::get('filter-apply-selections-to-results'),
                    fn($query) => $query->where($product_filtering)
                )
                ->distinct()
                ->orderByDesc('manufacture_year')
                ->pluck('manufacture_year');
        }

        return self::$data;
    }

    public function getIndexedResults(string $filter_type, int $limit, array $selections = []): Collection
    {
        return parent::getIndexedResults($filter_type, $limit, $selections)
            ->sortBy(fn($value) => (int)$value->filter_id);
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'hide_in_ui' => $this->hideInUi(),
            'has_load_more' => $this->hasLoadMore($collection),
            'results' => $collection,
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
            'is_range_maximum' => $this->isRangeMaximum(),
            'range_group' => $this->getRangeGroup()
        ];
    }

    public function getModel(): string
    {
        return '';
    }

    public function title(): string
    {
        return 'Manufacture Year';
    }

    public function modelSlug(Model $model): string
    {
        return '';
    }

    public function uiComponentType(): string
    {
        return 'range-filter';
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return 'Year From ' . $selection;
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/year-from-[0-9]+/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        preg_match('/year-from-([0-9]+)/', $selection, $matches);
        return $matches[1];
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return 'year-from-' . $selection;
    }

    protected function getRangeGroup(): string
    {
        return 'manufacture_year';
    }

    public function filterType(): string
    {
        return 'year_min';
    }
}
