<?php

namespace Mtc\MercuryDataModels;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Mtc\ContentManager\Traits\HasMedia;
use Mtc\MercuryDataModels\Factories\LabelFactory;
use OwenIt\Auditing\Contracts\Auditable;

class Label extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use HasFactory;
    use HasMedia;

    protected $table = 'labels';

    protected $fillable = [
        'name',
        'active',
        'type',
        'data',
        'content',
        'franchise_id',
        'category',
    ];

    protected $casts = [
        'data' => 'array',
        'active' => 'boolean',
    ];

    protected static function newFactory()
    {
        return LabelFactory::new();
    }

    public function vehicles(): BelongsToMany
    {
        return $this->belongsToMany(Vehicle::class, 'vehicle_labels');
    }

    public function offers(): BelongsToMany
    {
        return $this->belongsToMany(VehicleOffer::class, 'vehicle_offer_labels');
    }

    public function rules(): HasMany
    {
        return $this->HasMany(LabelRule::class, 'label_id')->orderBy('id');
    }

    /**
     * Sorting - customized from default due to views
     *
     * @param Builder $query
     * @param string|null $sortOption
     * @return Builder
     */
    public function scopeSetSortBy(Builder $query, ?string $sortOption): Builder
    {
        if (empty($sortOption)) {
            return $query;
        }

        if ($sortOption == 'franchise') {
            return $query->select('labels.*')
                ->leftJoin('franchises', 'franchise_id', '=', 'franchises.id')
                ->orderByRaw('franchises.name')
                ->orderByRaw('labels.name');
        }

        $direction = str_ends_with($sortOption, '_desc') ? 'desc' : 'asc';
        return $query->orderBy(str_replace('_desc', '', $sortOption), $direction);
    }
}
