<?php

namespace Mtc\MercuryDataModels\Traits;

use Carbon\Carbon;

trait VehicleAgeIdentifierFinder
{
    /**
     * Get the UK age identifier from the registration date.
     * Age identifier should be a 2 digit number.
     *
     * The year runs from March to March for registrations.
     *
     * If reg date between start of March and end of August,
     * age identifier is reg year in 2 digit format.
     *
     * If reg date between start of September and end of December,
     * age identifier is (reg year + 50) in 2 digit format.
     *
     * If reg date between start of January and end of March,
     * age identifier is (previous reg year + 50) in 2 digit format,
     * as the registration belongs to the previous year.
     *
     * @return int|string
     */
    public function getAgeIdentifier(?string $registration_date_string = ''): string|int
    {
        if (empty($registration_date_string)) {
            if (!empty($this->first_registration_date)) {
                $registration_date_string = $this->first_registration_date;
            } else {
                return '';
            }
        }

        $registration_date = Carbon::createFromFormat('Y-m-d H:i:s', $registration_date_string);

        if (empty($registration_date) || $registration_date->format('Y') < 2001) {
            return '';
        }

        $reg_year = (int)$registration_date->format('y');
        $reg_month = (int)$registration_date->format('m');

        if ($reg_month < 3) {
            // the identifier should relate to the previous year,
            // as the year for registrations runs March to March
            return ($reg_year - 1) + 50;
        }

        return $reg_month < 9 ? $reg_year : ($reg_year + 50);
    }
}
