<?php

namespace Mtc\MercuryDataModels\Filters;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\VehicleAttributeValue;

class StockStatusFilter extends IsFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $selections_to_search = [];

        // Stock statuses do not have a slug and a value, like taxonomies.
        // We need to support searches by slugs when the values may not contain e.g. dash characters.
        collect($selection)->each(function ($item) use (&$selections_to_search) {
            $selections_to_search[] = $item;
            $selections_to_search[] = str_replace('-', ' ', $item);
        });

        $query->whereHas('attributeValues', fn($query) => $query->where('slug', 'stock_status')
            ->whereIn('value', $selections_to_search));
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'hide_in_ui' => true,
            'has_load_more' => $this->hasLoadMore($collection),
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
            'results' => $collection
                ->filter(fn(Model $model) => !empty(trim($model[$this->getNameAttribute()])))
                ->map(fn(Model $model) => [
                    'id' => $model[$this->getNameAttribute()],
                    'name' => $model[$this->getNameAttribute()],
                    'count' => $model->result_count,
                    'slug' => $model[$this->getNameAttribute()]
                ]),
        ];
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return collect([]);
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return VehicleAttributeValue::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'Stock Status';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->value;
    }

    /**
     * Specify attribute on object that represents name
     *
     * @return string
     */
    public function getNameAttribute(bool $for_index = false): string
    {
        return 'value';
    }
}
