<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Closure;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Mtc\MercuryDataModels\Contracts\MinMaxFilter;
use Mtc\MercuryDataModels\Vehicle;

class MileageMaxFilter extends MileageMinFilter implements MinMaxFilter
{
    private ?int $min_value = null;

    public function applyFilter($query, array $selection = [])
    {
        $mileage_column = $this->mileageColumn();
        $query->where($mileage_column, '<=', $selection);
    }

    /**
     * Get available results of this filter type
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return $this->getData($product_filtering, $limit, $selections)
            ->filter(fn($range) => $this->isInBounds($range))
            ->map(fn ($range) => [
                'id' => $range,
                'name' => $this->getSelectionName($range)
            ])
            ->values();
    }

    protected function isRangeMaximum(): bool
    {
        return true;
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return 'Up To ' . number_format($selection) . ' ' . Settings::get('automotive-distance_measurement');
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/mileage-up-to-[0-9]+/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        preg_match('/mileage-up-to-([0-9]+)/', $selection, $matches);
        return $matches[1];
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return 'mileage-up-to-' . $selection;
    }

    public function filterType(): string
    {
        return 'mileage_max';
    }

    public function getOtherBound(): string
    {
        return 'mileage_min';
    }

    public function setOtherBoundary(?array $value = null): void
    {
        if (!empty($value)) {
            $this->min_value = $value[0];
        }
    }

    private function isInBounds($value): bool
    {
        if (empty($this->min_value)) {
            return true;
        }
        return $value >= $this->min_value;
    }
}
