<?php

namespace Mtc\MercuryDataModels;

use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Contracts\GlobalContent as GlobalContentModel;
use Mtc\MercuryDataModels\Contracts\ContentModel;
use Mtc\MercuryDataModels\Contracts\SearchableModel;
use Mtc\MercuryDataModels\Tools\UiUrlGenerator;
use OwenIt\Auditing\Contracts\Auditable;

class GlobalContent extends GlobalContentModel implements SearchableModel, Auditable, ContentModel
{
    use \OwenIt\Auditing\Auditable;

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'language',
        'name',
        'slug',
        'content',
        'element_id',
        'preview_image',
        'available_from',
        'available_to',
        'ui_component',
    ];

    /**
     * Cast attributes to certain types
     *
     * @var string[]
     */
    protected $casts = [
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'content' => 'array',
    ];

    protected static function boot()
    {
        parent::boot();
        self::creating(fn(self $content) => $content->fallbackUiComponent());
    }

    /**
     * Relationship with users that are managing this content element
     *
     * @return BelongsToMany
     */
    public function users(): BelongsToMany
    {
        $pivot_table = 'global_content_users';
        if (app()->runningUnitTests() === false) {
            $pivot_table = $this->getConnection()->getDatabaseName() . '.' . $pivot_table;
        }
        return $this->belongsToMany(Config::get('auth.providers.users.model'), $pivot_table, 'content_id');
    }

    /**
     * Search name text
     *
     * @return string
     */
    public function getSearchNameAttribute(): string
    {
        return $this->name;
    }

    /**
     * Search excerpt text
     *
     * @return string
     */
    public function getSearchExcerptAttribute(): string
    {
        return '';
    }

    /**
     * Search result icon
     *
     * @return string
     */
    public function getSearchIcon(): string
    {
        return 'pager';
    }

    /**
     * Route to viewing a vehicle as a part of search response
     *
     * @return string
     */
    public function getSearchResultRoute(): string
    {
        return UiUrlGenerator::make('manage-content/global/edit/' . $this->id, [], false);
    }

    /**
     * Set the ui_component value based on underlying content element if its not specified
     *
     * @return void
     */
    private function fallbackUiComponent()
    {
        if (empty($this->attributes['ui_component'])) {
            $this->ui_component = $this->contentElement()->first()?->ui_component;
        }
    }

    /**
     * Relationship with content element that defines structure of this element
     *
     * @return HasMany
     */
    public function subContent(): HasMany
    {
        return $this->hasMany(Config::get('pages.element_model'), 'element_id');
    }
}
