<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\Builder as QueryBuilder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\MercuryDataModels\Vehicle;

class ConsumptionFilter extends IndexedFilter implements CustomPatternFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder|QueryBuilder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selections = [])
    {
        $query->where(fn($priceQuery) => collect($selections)
            ->each(fn($selection) => $priceQuery->orWhere('mpg', '>=', $selection)));
    }

    /**
     * Get available results of this filter type
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        $min = floor(Vehicle::query()
            ->when(Settings::get('filter-apply-selections-to-results'), fn($query) => $query->where($product_filtering))
            ->min('mpg'));
        $max = ceil(Vehicle::query()
            ->when(Settings::get('filter-apply-selections-to-results'), fn($query) => $query->where($product_filtering))
            ->max('mpg'));

        $ranges = collect(Config::get('automotive.filter-ranges.mpg'));
        return $ranges
            // Up to max range including one step over current max
            ->filter(fn($range, $i) => $range < $max || ($i > 0 && $range > $max && $ranges[$i - 1] < $max))
            ->filter(fn($range, $i) => $range > $min
                || ($i < $ranges->count() - 1 && $range < $min && $ranges[$i + 1] > $min))
            ->map(fn($range, $i) => [
                'id' => $range,
                'value' => $range,
                'name' => __('filter::filter.mpg_range', [
                    'from' => number_format($range ?? 0),
                ]),

            ])
            ->values();
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return Vehicle::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return __('filter::filter.mpg_title');
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return '';
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'hide_in_ui' => $this->hideInUi(),
            'has_load_more' => $this->hasLoadMore($collection),
            'results' => $collection,
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
        ];
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/mpg-above-[0-9]+/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        preg_match('/mpg-above-([0-9]+)/', $selection, $matches);
        return $matches[1];
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return __('filter::filter.mpg_range', ['from' => $selection]);
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return 'mpg-above-' . $selection;
    }


    /**
     * Specify whether model should be re-indexed in the index
     * @return bool
     */
    public function reindexModel(): bool
    {
        return false;
    }

    /**
     * Whether this
     * @return bool
     */
    public function visibleByDefault(): bool
    {
        return true;
    }

    public function filterType(): string
    {
        return 'mpg';
    }
}
